% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fitIndependentCensoringModels.R
\name{fitIndepCens}
\alias{fitIndepCens}
\title{Fit Independent Censoring Models}
\usage{
fitIndepCens(
  resData,
  X,
  W,
  dist = c("Weibull", "lognormal"),
  start = NULL,
  n.iter = 20,
  bootstrap = TRUE,
  n.boot = 50,
  eps = 0.001
)
}
\arguments{
\item{resData}{Data matrix with three columns;  Z = the observed survival time, d1 = the censoring indicator of T
and  d2 =  the censoring indicator of C.}

\item{X}{Data matrix with covariates related to T.}

\item{W}{Data matrix with covariates related to C. First column of W should be ones.}

\item{dist}{The distribution to be used for the censoring time C. Only two distributions are allowed, i.e, Weibull
and lognormal distributions. With the value \code{"Weibull"} as the
default.}

\item{start}{Initial values for the finite dimensional parameters. If \code{start} is NULL, the initial values will be obtained
by fitting a Cox model for survival time T and a Weibull model for censoring time C.}

\item{n.iter}{Number of iterations; the default is \code{n.iter = 20}. The larger the number of iterations, the longer the computational time.}

\item{bootstrap}{A boolean indicating whether to compute bootstrap standard errors for making inferences.}

\item{n.boot}{Number of bootstrap samples to use in the estimation of bootstrap standard errors if \code{bootstrap = TRUE}. The default is n.boot = 50. But, higher
values  of \code{n.boot} are recommended for obtaining good estimates of bootstrap standard errors.}

\item{eps}{Convergence error. This is set by the user in such away that the desired convergence is met; the default is \code{eps = 1e-3}.}
}
\value{
This function returns a fit of independent censoring model; parameter estimates, estimate of the cumulative hazard function, bootstrap standard
errors for finite-dimensional parameters, the nonparametric cumulative hazard function, etc.
}
\description{
This function allows to estimate all model parameters under the assumption of independent censoring. First, estimates the cumulative hazard function, and
then at the second stage it estimates other model parameters assuming that the cumulative hazard is known.
}
\examples{
\donttest{

# Toy data example to illustrate implementation
n = 300
beta = c(0.5)
lambd = 0.35
eta = c(0.9,0.4)
X = cbind(rbinom(n,1,0.5))
W = cbind(rep(1,n),rbinom(n,1,0.5))
# generate dependency structure from Frank
frank.cop <- copula::frankCopula(param = 5,dim = 2)
U = copula::rCopula(n,frank.cop)
T1 = (-log(1-U[,1]))/(lambd*exp(X*beta))                  # Survival time'
T2 = (-log(1-U[,2]))^(1.1)*exp(W\%*\%eta)                   # Censoring time
A = runif(n,0,15)                                         # administrative censoring time
Z = pmin(T1,T2,A)
d1 = as.numeric(Z==T1)
d2 = as.numeric(Z==T2)
resData = data.frame("Z" = Z,"d1" = d1, "d2" = d2)     # should be data frame

# Fit independent censoring model

fitI <- fitIndepCens(resData = resData, X = X, W = W, bootstrap = FALSE)

# parameter estimates

fitI$parameterEstimates

# summary fit results
summary(fitI)

# plot cumulative hazard vs time

 plot(fitI$observedTime, fitI$cumhazardFunction, type = "l",xlab = "Time",
 ylab = "Estimated cumulative hazard function")
}

}
