% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/svem_optimize_random.R
\name{svem_optimize_random}
\alias{svem_optimize_random}
\title{Random-Search Optimizer with Goals, Weights, Optional CIs, and Diverse Candidates}
\usage{
svem_optimize_random(
  objects,
  goals,
  n = 50000,
  mixture_groups = NULL,
  debias = FALSE,
  agg = c("parms", "mean"),
  ci = TRUE,
  level = 0.95,
  top_frac = 0.01,
  k_candidates = 0,
  verbose = TRUE,
  combine = c("geom", "mean"),
  geom_floor = 1e-06,
  numeric_sampler = c("random", "maximin", "uniform")
)
}
\arguments{
\item{objects}{Named list of \code{svem_model} objects (from \code{SVEMnet()}).
List names must match the response names (left-hand sides) of the models.}

\item{goals}{Named list per response of the form
\code{list(goal = "max"|"min"|"target", weight = nonnegative number, target = number when goal = "target")}.
Weights are normalized to sum to one internally.}

\item{n}{Number of random samples to draw.}

\item{mixture_groups}{Optional mixture constraints forwarded to \code{svem_random_table_multi()}.
Each group may include \code{vars}, \code{lower}, \code{upper}, and \code{total}.}

\item{debias}{Logical; if \code{TRUE}, use debiased predictions for scoring where available.}

\item{agg}{Aggregation for point predictions, one of \code{"parms"} or \code{"mean"}.
This is passed to \code{predict.svem_model} when applicable.}

\item{ci}{Logical; if \code{TRUE}, compute percentile confidence intervals when available
via \code{predict(..., interval = TRUE)} or \code{predict_with_ci(...)}.}

\item{level}{Confidence level for percentile intervals (default \code{0.95}).}

\item{top_frac}{Fraction in \code{(0, 1]} of highest-score rows to cluster (default \code{0.01}).}

\item{k_candidates}{Integer number of diverse candidates (medoids) to return (default \code{0}).
If \code{0}, no clustering is performed.}

\item{verbose}{Logical; print a compact summary of the run and results.}

\item{combine}{How to combine normalized per-response scores. Use \code{"geom"} for
weighted geometric mean (default) or \code{"mean"} for weighted arithmetic mean.}

\item{geom_floor}{Small positive floor in \code{(0,1)} used only for \code{combine = "geom"}.
Each normalized score z in \code{[0,1]} is mapped to \code{z_adj = geom_floor + (1 - geom_floor) * z}
before taking logs to avoid \code{log(0)}. Default \code{1e-6}.}

\item{numeric_sampler}{Sampler for non-mixture numeric predictors passed to
\code{svem_random_table_multi}. One of \code{"random"} (default), \code{"maximin"}, or \code{"uniform"}.
"random" uses \code{lhs::randomLHS} when available, otherwise plain \code{runif}.}
}
\value{
A list with:
\describe{
\item{best_idx}{Row index of the selected best design in the sampled table.}
\item{best_x}{Predictors at the best design.}
\item{best_pred}{Named numeric vector of predicted responses at \code{best_x}.}
\item{best_ci}{Data frame of percentile limits when \code{ci = TRUE}; otherwise \code{NULL}.}
\item{candidates}{Data frame of \code{k_candidates} diverse candidates (medoids; existing rows) with
predictors, predictions, optional CIs, and score; \code{NULL} if \code{k_candidates = 0}.}
\item{score_table}{Sampled table with response columns, normalized contributions,
weighted contributions, optional log-weighted terms (when \code{combine = "geom"}), and final score.}
\item{weights}{Normalized weights used in scoring.}
\item{goals}{Tidy data frame describing each response goal, weight, and target.}
}
}
\description{
Draws random points via \code{svem_random_table_multi}, scores them using user
goals and weights, returns the best design point, and optionally proposes
\code{k_candidates} diverse high-score candidates by clustering the top fraction
of rows with Gower distance and PAM medoids. Medoids are representative
existing rows, so proposed candidates are guaranteed to be feasible under all
sampling and mixture constraints.
}
\section{Scoring}{

Each response is normalized on the sampled range with \code{normalize01} mapping to \code{[0,1]}:
\itemize{
\item \code{goal = "max"} uses \code{normalize01(y)}.
\item \code{goal = "min"} uses \code{normalize01(-y)}.
\item \code{goal = "target"} uses \code{normalize01(-abs(y - target))}.
}
Then the normalized values are combined across responses:
\itemize{
\item \code{combine = "mean"}: weighted arithmetic mean, computed as \code{sum_r w_r * z_r}.
\item \code{combine = "geom"}: weighted geometric mean, computed as
\code{exp( sum_r w_r * log(z_adj_r) )} with \code{z_adj_r = geom_floor + (1 - geom_floor) * z_r}.
}
}

\section{Diverse candidates}{

We take the top \code{top_frac} fraction by score, compute Gower distances on predictors,
and run PAM to get medoids. Returning medoids rather than centroids ensures each
candidate corresponds to an actual sampled setting that satisfies constraints.
}

\examples{
\donttest{
set.seed(1)
n  <- 120
X1 <- runif(n); X2 <- runif(n)
F  <- factor(sample(c("lo","hi"), n, TRUE))
y1 <- 1 + 1.5*X1 - 0.8*X2 + 0.4*(F=="hi") + rnorm(n, 0, 0.2)
y2 <- 0.7 + 0.4*X1 + 0.4*X2 - 0.2*(F=="hi") + rnorm(n, 0, 0.2)
dat <- data.frame(y1, y2, X1, X2, F)

m1 <- SVEMnet(y1 ~ X1 + X2 + F, dat, nBoot = 30)
m2 <- SVEMnet(y2 ~ X1 + X2 + F, dat, nBoot = 30)
objs <- list(y1 = m1, y2 = m2)

goals <- list(
  y1 = list(goal = "max",    weight = 0.6),
  y2 = list(goal = "target", weight = 0.4, target = 0.9)
)

# Default combine = "geom" and numeric_sampler = "random"
out <- svem_optimize_random(
  objects      = objs,
  goals        = goals,
  n            = 5000,
  agg          = "mean",
  debias       = FALSE,
  ci           = TRUE,
  level        = 0.95,
  k_candidates = 5,
  top_frac     = 0.02,
  numeric_sampler = "random",
  verbose      = TRUE
)
out$best_x; head(out$candidates)

# Use arithmetic mean instead, with fastest uniform sampler
out_mean <- svem_optimize_random(
  objects      = objs,
  goals        = goals,
  n            = 5000,
  combine      = "mean",
  numeric_sampler = "uniform",
  verbose      = FALSE
)

# Mixture-constrained lipid example (composition sums to 1)
data(lipid_screen)
spec <- bigexp_terms(
  Potency ~ PEG + Helper + Ionizable + Cholesterol +
    Ionizable_Lipid_Type + N_P_ratio + flow_rate,
  data = lipid_screen, factorial_order = 3
)
fP <- bigexp_formula(spec, "Potency")
fS <- bigexp_formula(spec, "Size")
fD <- bigexp_formula(spec, "PDI")
mP <- SVEMnet(fP, lipid_screen, nBoot = 40)
mS <- SVEMnet(fS, lipid_screen, nBoot = 40)
mD <- SVEMnet(fD, lipid_screen, nBoot = 40)
objs2 <- list(Potency = mP, Size = mS, PDI = mD)

goals2 <- list(
  Potency = list(goal = "max", weight = 0.7),
  Size    = list(goal = "min", weight = 0.2),
  PDI     = list(goal = "min", weight = 0.1)
)

mixL <- list(list(
  vars  = c("Cholesterol","PEG","Ionizable","Helper"),
  lower = c(0.10, 0.01, 0.10, 0.10),
  upper = c(0.60, 0.05, 0.60, 0.60),
  total = 1
))

opt <- svem_optimize_random(
  objects        = objs2,
  goals          = goals2,
  n              = 8000,
  mixture_groups = mixL,
  agg            = "mean",
  debias         = FALSE,
  ci             = TRUE,
  level          = 0.95,
  k_candidates   = 5,
  top_frac       = 0.01,
  numeric_sampler = "random",
  verbose        = TRUE
)
opt$best_x; head(opt$candidates)
}

}
