/*
 * This file is part of ReporteRs
 * Copyright (c) 2014, David Gohel All rights reserved.
 * This program is licensed under the GNU GENERAL PUBLIC LICENSE V3.
 * You may obtain a copy of the License at :
 * http://www.gnu.org/licenses/gpl.html
 */

package org.lysis.reporters.lists;

import java.math.BigInteger;

import org.docx4j.dml.CTTextAutonumberBullet;
import org.docx4j.dml.CTTextCharBullet;
import org.docx4j.dml.CTTextParagraphProperties;
import org.docx4j.dml.STTextAutonumberScheme;
import org.docx4j.dml.TextFont;
import org.docx4j.wml.Jc;
import org.docx4j.wml.JcEnumeration;
import org.docx4j.wml.Lvl;
import org.docx4j.wml.Lvl.LvlText;
import org.docx4j.wml.Lvl.Start;
import org.docx4j.wml.NumFmt;
import org.docx4j.wml.NumberFormat;
import org.docx4j.wml.PPr;
import org.docx4j.wml.PPrBase.Ind;

public class LevelDefinition {
	private double indent_left;
	private double indent_hanging;
	private String textalign;
	private String format;
	private String pattern;
	private int level;
	
	private String cssClassName;
	
	public LevelDefinition(double indent_left, double indent_hanging,
			String textalign, String format, String pattern, int level) {
		super();
		this.indent_left = indent_left;
		this.indent_hanging = indent_hanging;
		this.textalign = textalign;
		this.format = format;
		this.pattern = pattern;
		this.level = level;
	}

	public String getCssClassName() {
		return cssClassName;
	}

	public void setCssClassName(String cssClassName) {
		this.cssClassName = cssClassName;
	}

	public double getIndent_left() {
		return indent_left;
	}

	public void setIndent_left(double indent_left) {
		this.indent_left = indent_left;
	}

	public int getLevel() {
		return level;
	}

	public void setLevel(int level) {
		this.level = level;
	}

	public double getIndent_hanging() {
		return indent_hanging;
	}

	public void setIndent_hanging(double indent_hanging) {
		this.indent_hanging = indent_hanging;
	}

	public String getAlignment() {
		return textalign;
	}

	public void setAlignment(String textalign) {
		this.textalign = textalign;
	}

	public String getFormat() {
		return format;
	}

	public void setFormat(String format) {
		this.format = format;
	}

	public String getPattern() {
		return pattern;
	}

	public void setPattern(String pattern) {
		this.pattern = pattern;
	}

	private STTextAutonumberScheme get_pptx_NumberFormat(){

		if( format.equals("decimal")){
			return STTextAutonumberScheme.ARABIC_PERIOD;
		} else if( format.equals("upperRoman")){
			return STTextAutonumberScheme.ROMAN_UC_PAREN_BOTH;
		} else if( format.equals("lowerRoman")){
			return STTextAutonumberScheme.ROMAN_LC_PERIOD;
		} else if( format.equals("upperLetter")){
			return STTextAutonumberScheme.ALPHA_UC_PERIOD;
		} else if( format.equals("lowerLetter")){
			return STTextAutonumberScheme.ALPHA_LC_PERIOD;
		} else {
			return STTextAutonumberScheme.ARABIC_PERIOD;
		} 
	}

	
	public void set_pptx_scheme(CTTextParagraphProperties parProperties, int startNumbering){
		
		int mar_left = (int)( (getIndent_left()+getIndent_hanging()) * 914400);
		parProperties.setMarL(mar_left);

		if( getLevel() > 0 ){
			parProperties.setLvl( getLevel() );
		}

		parProperties.setIndent((int) (-getIndent_hanging() * 914400) );
		
		if( isOrdered() ){
			CTTextAutonumberBullet autonum = new CTTextAutonumberBullet();
			autonum.setType(get_pptx_NumberFormat());
			parProperties.setBuAutoNum(autonum);

    		if( startNumbering > 1 && getLevel() < 1 ) autonum.setStartAt(startNumbering);

    	} else if( isUnordered() ){
    		CTTextCharBullet textBullet = new CTTextCharBullet();
    		Integer pitchFamily=0;
    		Integer charset=0;
    		TextFont tf = new TextFont();
    		if( format.equals("square")){
    			pitchFamily = 2;
    			charset = 2;
    			tf.setTypeface("Wingdings");
        		textBullet.setChar("§");
    		} else if( format.equals("circle")){
    			pitchFamily = 49;
    			charset = 0;
    			tf.setTypeface("Courier New");
        		textBullet.setChar("o");
    		} else {
    			pitchFamily = 34;
    			charset = 0;
    			tf.setTypeface("Arial");
        		textBullet.setChar("\u2022");
    		}
    		tf.setCharset(charset.byteValue());
			tf.setPitchFamily(pitchFamily.byteValue());
    		parProperties.setBuFont(tf );
    		parProperties.setBuChar(textBullet);
    	}
	}
	

	public boolean isOrdered(){
		if( format.equals("decimal") || format.equals("upperRoman") || 
				format.equals("lowerRoman") || format.equals("upperLetter") || 
				format.equals("lowerLetter") ){
			return true;
		} else {
			return false;
		}
	}
	public boolean isUnordered(){
		if( format.equals("circle") || format.equals("disc") || format.equals("square") ){
			return true;
		} else {
			return false;
		}
	}
	private String getCSSNumberFormat(){

		if( format.equals("decimal")){
			return ", decimal";
		} else if( format.equals("upperRoman")){
			return ", upper-roman";
		} else if( format.equals("lowerRoman")){
			return ", lower-roman";
		} else if( format.equals("upperLetter")){
			return ", upper-alpha";
		} else if( format.equals("lowerLetter")){
			return ", lower-alpha";
		} else if( format.equals("circle")){
			return ", circle";
		} else if( format.equals("disc")){
			return ", disc";
		} else if( format.equals("square")){
			return ", square";
		} else {
			return "circle";
		} 
	}
	private NumberFormat getNumberFormat(){

		if( format.equals("decimal")){
			return NumberFormat.DECIMAL;
		} else if( format.equals("upperRoman")){
			return NumberFormat.UPPER_ROMAN;
		} else if( format.equals("lowerRoman")){
			return NumberFormat.LOWER_ROMAN;
		} else if( format.equals("upperLetter")){
			return NumberFormat.UPPER_LETTER;
		} else if( format.equals("lowerLetter")){
			return NumberFormat.LOWER_LETTER;
		} else {
			return NumberFormat.BULLET;
		} 
	}
	
	private String getDocxPattern(){

		if( format.equals("decimal")){
			return pattern;
		} else if( format.equals("upperRoman")){
			return pattern;
		} else if( format.equals("lowerRoman")){
			return pattern;
		} else if( format.equals("upperLetter")){
			return pattern;
		} else if( format.equals("lowerLetter")){
			return pattern;
		} else if( format.equals("disc")){
			return "\u25CF ";
		} else if( format.equals("circle")){
			return "\u25CB ";
		} else if( format.equals("square")){
			return "\u220E ";
		} else {
			return "\u25CF ";
		} 
	}
	
	public Lvl getLvl() throws Exception {

		Lvl lvl = new Lvl();
		lvl.setIlvl(BigInteger.valueOf( level ));
		
		Jc alignment = new Jc();
		if( textalign.equals("left")) alignment.setVal(JcEnumeration.LEFT);
		else if( textalign.equals("center")) alignment.setVal(JcEnumeration.CENTER);
		else if( textalign.equals("right")) alignment.setVal(JcEnumeration.RIGHT);
		else if( textalign.equals("justify")) alignment.setVal(JcEnumeration.BOTH);
		lvl.setLvlJc(alignment);
		
		PPr parProperties = new PPr();
		
		Ind ind = new Ind();
		ind.setLeft(BigInteger.valueOf( (int)( (indent_left + indent_hanging) * 1440)  ));
		ind.setHanging(BigInteger.valueOf((int)(indent_hanging * 1440)));
		parProperties.setInd(ind);
		lvl.setPPr(parProperties);
		
		NumFmt nf = new NumFmt();
		nf.setVal(getNumberFormat());		
		lvl.setNumFmt(nf);
		LvlText lt = new LvlText();
		lt.setVal(getDocxPattern());
		lvl.setLvlText(lt);
		
		Start start = new Start();
		start.setVal(BigInteger.valueOf(1));
		lvl.setStart(start);
		
		return lvl;
	}
	
	public String getCounterName(int list_id){
		String CSSSuffix = list_id + "-" + (level+1);
		return "counter-" + CSSSuffix;
	}
	
	public int getPaddingLeft(){
		int dtbefore = (int)(indent_left*72);
		int dtpaddingleft = dtbefore + (int)(indent_hanging*72);
		return dtpaddingleft;
	}
	public String getHTMLCSS(int list_id ) {
		String CSSSuffix = list_id + "-" + (level+1);
		
		String counterDefExtraArg = getCSSNumberFormat();
		String CounterName = "counter-" + CSSSuffix;

		String CSSClassName = "";
		String CSSCounterIncrement = "";
		String CSSListformat = "";
		if( isUnordered() ){
			CSSClassName = "ul-" + CSSSuffix;
			CSSCounterIncrement = "";
		} else {
			CSSClassName = "ol-" + CSSSuffix;
			CSSCounterIncrement = "counter-increment: " +  CounterName + ";";
		} 
		
		String tempcss = pattern.replaceAll("(%){1}([0-9]){1}", "counter(counter-"+ list_id + "-$2" + counterDefExtraArg + ") ");
		CSSListformat = "content: " + "\"" + tempcss.replaceAll("(counter\\(([a-z0-9\\-\\,\\s])+\\)){1}", "\" $1 \"" ) + "\";";
		int dtbefore = (int)(indent_left*72);
		String out ="";
		out += "." + CSSClassName + " dt::before{" + CSSCounterIncrement +  CSSListformat
				+ "position: absolute;left:" + dtbefore + "px" + ";" 
			+ "}";
		out += "." + CSSClassName + " dt {position: relative;font-weight: normal;}";
		return out;
	}

}
