// Updated by CRAN team from
// https://github.com/blob79/phonetik/tree/master/libphonet/src/main/native
// to resolve encoding issues.
/*
 * phonet.h
 * --------
 *
 * Phonetic rules for "phonet.c".
 *
 * Copyright (c):
 * 1999-2007:  Joerg MICHAEL, Adalbert-Stifter-Str. 11, 30655 Hannover, Germany
 *     and
 * (version 1.0) 1999:  Heise Verlag, Helstorfer Str. 7, 30625 Hannover, Germany
 *
 * SCCS: @(#) phonet.h  1.4.2  2007-08-27
 *
 * This program is subject to the GNU Lesser General Public License (LGPL)
 * (formerly known as GNU Library General Public Licence)
 * as published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Actually, the LGPL is __less__ restrictive than the better known GNU General
 * Public License (GPL). See the GNU Library General Public License or the file
 * LIB_GPLP.TXT for more details and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * There is one important restriction: If you modify this program in any way
 * (e.g. add or change phonetic rules or modify the underlying logic or
 * translate this program into another programming language), you must also
 * release the changes under the terms of the LGPL.
 * That means you have to give out the source code to your changes,
 * and a very good way to do so is mailing them to the address given below.
 * I think this is the best way to promote further development and use
 * of this software.
 *
 * If you have any remarks, feel free to e-mail to:
 *     ct@ct.heise.de
 *
 * The author's email address is:
 *    astro.joerg@googlemail.com
 */


#ifndef _PHONET_H_
#define _PHONET_H_
//#pragma GCC diagnostic ignored "-Wunused-function"

#include <stdio.h>
#include <stdlib.h>

#include "ph_ext.h"
#include <R.h>

#define  PHONET_VERSION  "version 1.4.2, 2007-08-27"

/****  we assume 1 byte chars  ****/
#define  HASH_COUNT    256

/****  macro to make the rules easier readable  ****/
#define  null  NULL


/****  end marker for string table  ****/
static char PHONET_END[] = "";

static char *phonet_language = "unknown";
static int  *phonet_init = NULL;
static int  *phonet_hash = NULL;
static int  (*phonet_hash_1)[26][28] = NULL;   /***  these are pointers  ***/
static int  (*phonet_hash_2)[26][28] = NULL;   /***  to  array[26][28]   ***/
static char **phonet_rules = NULL;


/****  Syntax for phonetic rules:               ****/
/****    <search_string>  <1st_rule> <2nd_rule> ****/
/****  Syntax for search strings:               ****/
/****    <word> [<->..] [<] [<0-9>] [^[^]] [$]  ****/

/****  Remarks:                                 ****/
/****  a) All phonetic rules must be written    ****/
/****     in upper case.                        ****/
/****  b) The end of <word> may contain as an   ****/
/****     optional simple regular expression    ****/
/****     one array of letters (or umlaut's)    ****/
/****     enclosed in '(' and ')'.              ****/
/****  c) Rules with a '<' demand that the      ****/
/****     replacement string may not be longer  ****/
/****     than the search string.               ****/
/****  d) The placement of rules determines     ****/
/****     their priority. Therefore, the rules  ****/
/****     for "SH" must be placed before the    ****/
/****     rules for "S"                         ****/
/****     (otherwise, a conversion error will   ****/
/****      occur for "SH").                     ****/



#ifdef PHONET_ENGLISH

/*
 *
 * UNDER CONSTRUCTION
 * (no rules available at this time)
 *
 */

#endif    /****  PHONET_ENGLISH  ****/




#ifdef PHONET_NO_LANGUAGE

static char *phonet_language_no_language = "no language";
static int  phonet_init_no_language = 0;
static int  phonet_hash_no_language [HASH_COUNT];
static int  phonet_hash_no_language_1 [26][28];
static int  phonet_hash_no_language_2 [26][28];

/****  Phonetic rules for "no language"   ****/
/****  (i.e. umlauts will be "expanded")  ****/

static char *phonet_rules_no_language[] =
  {
/****  separator chars  ****/
    "´",               " ",       " ",
    "'",               " ",       " ",
    "`$",              "",        "",
    "\"",              " ",       " ",
    ",",               ",",       ",",
    ";",               ",",       ",",
    "-",               " ",       " ",
    " ",               " ",       " ",
    ".",               ".",       ".",
    ":",               ".",       ".",
/****  German umlauts  ****/
    "Ä",               "AE",      "AE",
    "Ö",               "OE",      "OE",
    "Ü",               "UE",      "UE",
    "ß",               "S",       "S",
/****  international umlauts  ****/
    "À",               "A",       "A",
    "Á",               "A",       "A",
    "Â",               "A",       "A",
    "Ã",               "A",       "A",
    "Å",               "A",       "A",
    "Æ",               "AE",      "AE",
    "Ç",               "C",       "C",
    "Ð",               "DJ",      "DJ",
    "È",               "E",       "E",
    "É",               "E",       "E",
    "Ê",               "E",       "E",
    "Ë",               "E",       "E",
    "Ì",               "I",       "I",
    "Í",               "I",       "I",
    "Î",               "I",       "I",
    "Ï",               "I",       "I",
    "Ñ",               "NH",      "NH",
    "Ò",               "O",       "O",
    "Ó",               "O",       "O",
    "Ô",               "O",       "O",
    "Õ",               "O",       "O",
    "Œ",               "OE",      "OE",
    "Ø",               "OE",      "OE",
    "Š",               "SH",      "SH",
    "Þ",               "TH",      "TH",
    "Ù",               "U",       "U",
    "Ú",               "U",       "U",
    "Û",               "U",       "U",
    "Ý",               "Y",       "Y",
    "Ÿ",               "Y",       "Y",
/****  "normal" letters (A-Z)  ****/
    "MC^",             "MAC",     "MAC",
    "MC^",             "MAC",     "MAC",
    "M´^",             "MAC",     "MAC",
    "M'^",             "MAC",     "MAC",
    "O´^",             "O",       "O",
    "O'^",             "O",       "O",
    "VAN DEN ^",      "VANDEN",   "VANDEN",
    PHONET_END,      PHONET_END,  PHONET_END
  };

#endif    /****  PHONET_NO_LANGUAGE  ****/




#ifdef PHONET_GERMAN

static char *phonet_language_german = "German";
static int  phonet_init_german = 0;
static int  phonet_hash_german [HASH_COUNT];
static int  phonet_hash_german_1 [26][28];
static int  phonet_hash_german_2 [26][28];

/****  Phonetic rules for German.     ****/
/****  First Rules:                   ****/
/****    The letters Ä,ß,C,J,Q,V,Y    ****/
/****    will "disappear".            ****/
/****  Second Rules (less accurate):  ****/
/****    The letters Ä,Ü,ß,C,D,       ****/
/****    G,H,J,O,P,Q,S,V,W,X,Y        ****/
/****    will "disappear".            ****/

static char *phonet_rules_german[] =
  {
/****  separator chars  ****/
    "´",               " ",       " ",
    "'",               " ",       " ",
    "`$",              "",        "",
    "\"",              " ",       " ",
    ",",               " ",       " ",
    ";",               " ",       " ",
    "-",               " ",       " ",
    " ",               " ",       " ",
    ".",               ".",       ".",
    ":",               ".",       ".",
/****  German umlauts  ****/
    "ÄE",              "E",       "E",
    "ÄU<",             "EU",      "EU",
    "ÄV(AEOU)-<",      "EW",      null,
    "Ä$",              "Ä",       null,
    "Ä<",              null,      "E",
    "Ä",               "E",       null,
    "ÖE",              "Ö",       "Ö",
    "ÖU",              "Ö",       "Ö",
    "ÖVER--<",         "ÖW",      null,
    "ÖV(AOU)-",        "ÖW",      null,
    "ÜBEL(GNRW)-^^",   "ÜBL ",    "IBL ",
    "ÜBER^^",          "ÜBA",     "IBA",
    "ÜE",              "Ü",       "I",
    "ÜVER--<",         "ÜW",      null,
    "ÜV(AOU)-",        "ÜW",      null,
    "Ü",               null,      "I",
    "ßCH<",            null,      "Z",
    "ß<",              "S",       "Z",
/****  international umlauts  ****/
    "À<",              "A",       "A",
    "Á<",              "A",       "A",
    "Â<",              "A",       "A",
    "Ã<",              "A",       "A",
    "Å<",              "A",       "A",
    "ÆER-",            "E",       "E",
    "ÆU<",             "EU",      "EU",
    "ÆV(AEOU)-<",      "EW",      null,
    "Æ$",              "Ä",       null,
    "Æ<",              null,      "E",
    "Æ",               "E",       null,
    "Ç",               "Z",       "Z",
    "ÐÐ-",             "",        "",
    "Ð",               "DI",      "TI",
    "È<",              "E",       "E",
    "É<",              "E",       "E",
    "Ê<",              "E",       "E",
    "Ë",               "E",       "E",
    "Ì<",              "I",       "I",
    "Í<",              "I",       "I",
    "Î<",              "I",       "I",
    "Ï",               "I",       "I",
    "ÑÑ-",             "",        "",
    "Ñ",               "NI",      "NI",
    "Ò<",              "O",       "U",
    "Ó<",              "O",       "U",
    "Ô<",              "O",       "U",
    "Õ<",              "O",       "U",
    "Œ<",              "Ö",       "Ö",
    "Ø(IJY)-<",        "E",       "E",
    "Ø<",              "Ö",       "Ö",
    "Š",               "SH",      "Z",
    "Þ",               "T",       "T",
    "Ù<",              "U",       "U",
    "Ú<",              "U",       "U",
    "Û<",              "U",       "U",
    "Ý<",              "I",       "I",
    "Ÿ<",              "I",       "I",
/****  "normal" letters (A-Z)  ****/
    "ABELLE$",         "ABL",     "ABL",
    "ABELL$",          "ABL",     "ABL",
    "ABIENNE$",        "ABIN",    "ABIN",
    "ACHME---^",       "ACH",     "AK",
    "ACEY$",           "AZI",     "AZI",
    "ADV",             "ATW",     null,
    "AEGL-",           "EK",      null,
    "AEU<",            "EU",      "EU",
    "AE2",             "E",       "E",
    "AFTRAUBEN------", "AFT ",    "AFT ",
    "AGL-1",           "AK",      null,
    "AGNI-^",          "AKN",     "AKN",
    "AGNIE-",          "ANI",     "ANI",
    "AGN(AEOU)-$",     "ANI",     "ANI",
    "AH(AIOÖUÜY)-",    "AH",      null,
    "AIA2",            "AIA",     "AIA",
    "AIE$",            "E",       "E",
    "AILL(EOU)-",      "ALI",     "ALI",
    "AINE$",           "EN",      "EN",
    "AIRE$",           "ER",      "ER",
    "AIR-",            "E",       "E",
    "AISE$",           "ES",      "EZ",
    "AISSANCE$",       "ESANS",   "EZANZ",
    "AISSE$",          "ES",      "EZ",
    "AIX$",            "EX",      "EX",
    "AJ(AÄEÈÉÊIOÖUÜ)--", "A",     "A",
    "AKTIE",           "AXIE",    "AXIE",
    "AKTUEL",          "AKTUEL",  null,
    "ALOI^",           "ALOI",    "ALUI",  /** Do NOT merge these rules  **/
    "ALOY^",           "ALOI",    "ALUI",  /** (needed by "check_rules") **/
    "AMATEU(RS)-",     "AMATÖ",   "ANATÖ",
    "ANCH(OEI)-",      "ANSH",    "ANZ",
    "ANDERGEGANG----", "ANDA GE", "ANTA KE",
    "ANDERGEHE----",   "ANDA ",   "ANTA ",
    "ANDERGESETZ----", "ANDA GE", "ANTA KE",
    "ANDERGING----",   "ANDA ",   "ANTA ",
    "ANDERSETZ(ET)-----", "ANDA ", "ANTA ",
    "ANDERZUGEHE----", "ANDA ZU ", "ANTA ZU ",
    "ANDERZUSETZE-----", "ANDA ZU ", "ANTA ZU ",
    "ANER(BKO)---^^",  "AN",      null,
    "ANHAND---^$",     "AN H",    "AN ",
    "ANH(AÄEIOÖUÜY)--^^", "AN",   null,
    "ANIELLE$",        "ANIEL",   "ANIL",
    "ANIEL",           "ANIEL",   null,
    "ANSTELLE----^$",  "AN ST",   "AN ZT",
    "ANTI^^",          "ANTI",    "ANTI",
    "ANVER^^",         "ANFA",    "ANFA",
    "ATIA$",           "ATIA",    "ATIA",
    "ATIA(NS)--",      "ATI",     "ATI",
    "ATI(AÄOÖUÜ)-",    "AZI",     "AZI",
    "AUAU--",          "",        "",
    "AUERE$",          "AUERE",   null,
    "AUERE(NS)-$",     "AUERE",   null,
    "AUERE(AIOUY)--",  "AUER",    null,
    "AUER(AÄIOÖUÜY)-", "AUER",    null,
    "AUER<",           "AUA",     "AUA",
    "AUF^^",           "AUF",     "AUF",
    "AULT$",           "O",       "U",
    "AUR(BCDFGKLMNQSTVWZ)-", "AUA", "AUA",
    "AUR$",            "AUA",     "AUA",
    "AUSSE$",          "OS",      "UZ",
    "AUS(ST)-^",       "AUS",     "AUS",
    "AUS^^",           "AUS",     "AUS",
    "AUTOFAHR----",    "AUTO ",   "AUTU ",
    "AUTO^^",          "AUTO",    "AUTU",
    "AUX(IY)-",        "AUX",     "AUX",
    "AUX",             "O",       "U",
    "AU",              "AU",      "AU",
    "AVER--<",         "AW",      null,
    "AVIER$",          "AWIE",    "AFIE",
    "AV(EÈÉÊI)-^",     "AW",      null,
    "AV(AOU)-",        "AW",      null,
    "AYRE$",           "EIRE",    "EIRE",
    "AYRE(NS)-$",      "EIRE",    "EIRE",
    "AYRE(AIOUY)--",   "EIR",     "EIR",
    "AYR(AÄIOÖUÜY)-",  "EIR",     "EIR",
    "AYR<",            "EIA",     "EIA",
    "AYER--<",         "EI",      "EI",
    "AY(AÄEIOÖUÜY)--", "A",       "A",
    "AË",              "E",       "E",
    "A(IJY)<",         "EI",      "EI",
    "BABY^$",          "BEBI",    "BEBI",
    "BAB(IY)^",        "BEBI",    "BEBI",
    "BEAU^$",          "BO",      null,
    "BEA(BCMNRU)-^",   "BEA",     "BEA",
    "BEAT(AEIMORU)-^", "BEAT",    "BEAT",
    "BEE$",            "BI",      "BI",
    "BEIGE^$",         "BESH",    "BEZ",
    "BENOIT--",        "BENO",    "BENU",
    "BER(DT)-",        "BER",     null,
    "BERN(DT)-",       "BERN",    null,
    "BE(LMNRST)-^",    "BE",      "BE",
    "BETTE$",          "BET",     "BET",
    "BEVOR^$",         "BEFOR",   null,
    "BIC$",            "BIZ",     "BIZ",
    "BOWL(EI)-",       "BOL",     "BUL",
    "BP(AÄEÈÉÊIÌÍÎOÖRUÜY)-", "B", "B",
    "BRINGEND-----^",  "BRI",     "BRI",
    "BRINGEND-----",   " BRI",    " BRI",
    "BROW(NS)-",       "BRAU",    "BRAU",
    "BUDGET7",         "BÜGE",    "BIKE",
    "BUFFET7",         "BÜFE",    "BIFE",
    "BYLLE$",          "BILE",    "BILE",
    "BYLL$",           "BIL",     "BIL",
    "BYPA--^",         "BEI",     "BEI",
    "BYTE<",           "BEIT",    "BEIT",
    "BY9^",            "BÜ",      null,
    "B(SßZ)$",         "BS",      null,
    "CACH(EI)-^",      "KESH",    "KEZ",
    "CAE--",           "Z",       "Z",
    "CA(IY)$",         "ZEI",     "ZEI",
    "CE(EIJUY)--",     "Z",       "Z",
    "CENT<",           "ZENT",    "ZENT",
    "CERST(EI)----^",  "KE",      "KE",
    "CER$",            "ZA",      "ZA",
    "CE3",             "ZE",      "ZE",
    "CH'S$",           "X",       "X",
    "CH´S$",           "X",       "X",
    "CHAO(ST)-",       "KAO",     "KAU",
    "CHAMPIO-^",       "SHEMPI",  "ZENBI",
    "CHAR(AI)-^",      "KAR",     "KAR",
    "CHAU(CDFSVWXZ)-", "SHO",     "ZU",
    "CHÄ(CF)-",        "SHE",     "ZE",
    "CHE(CF)-",        "SHE",     "ZE",
    "CHEM-^",          "KE",      "KE",    /**  or: "CHE", "KE" **/
    "CHEQUE<",         "SHEK",    "ZEK",
    "CHI(CFGPVW)-",    "SHI",     "ZI",
    "CH(AEUY)-<^",     "SH",      "Z",
    "CHK-",            "",        "",
    "CHO(CKPS)-^",     "SHO",     "ZU",
    "CHRIS-",          "KRI",     null,
    "CHRO-",           "KR",      null,
    "CH(LOR)-<^",      "K",       "K",
    "CHST-",           "X",       "X",
    "CH(SßXZ)3",       "X",       "X",
    "CHTNI-3",         "CHN",     "KN",
    "CH^",             "K",       "K",     /**  or: "CH", "K"  **/
    "CH",              "CH",      "K",
    "CIC$",            "ZIZ",     "ZIZ",
    "CIENCEFICT----",  "EIENS ",  "EIENZ ",
    "CIENCE$",         "EIENS",   "EIENZ",
    "CIER$",           "ZIE",     "ZIE",
    "CYB-^",           "ZEI",     "ZEI",
    "CY9^",            "ZÜ",      "ZI",
    "C(IJY)-<3",       "Z",       "Z",
    "CLOWN-",          "KLAU",    "KLAU",
    "CCH",             "Z",       "Z",
    "CCE-",            "X",       "X",
    "C(CK)-",          "",        "",
    "CLAUDET---",      "KLO",     "KLU",
    "CLAUDINE^$",      "KLODIN",  "KLUTIN",
    "COACH",           "KOSH",    "KUZ",
    "COLE$",           "KOL",     "KUL",
    "COUCH",           "KAUSH",   "KAUZ",
    "COW",             "KAU",     "KAU",
    "CQUES$",          "K",       "K",
    "CQUE",            "K",       "K",
    "CRASH--9",        "KRE",     "KRE",
    "CREAT-^",         "KREA",    "KREA",
    "CST",             "XT",      "XT",
    "CS<^",            "Z",       "Z",
    "C(SßX)",          "X",       "X",
    "CT'S$",           "X",       "X",
    "CT(SßXZ)",        "X",       "X",
    "CZ<",             "Z",       "Z",
    "C(ÈÉÊÌÍÎÝ)3",     "Z",       "Z",
    "C.^",             "C.",      "C.",
    "CÄ-",             "Z",       "Z",
    "CÜ$",             "ZÜ",      "ZI",
    "C'S$",            "X",       "X",
    "C<",              "K",       "K",
    "DAHER^$",         "DAHER",   null,
    "DARAUFFOLGE-----", "DARAUF ", "TARAUF ",
    "DAVO(NR)-^$",     "DAFO",    "TAFU",
    "DD(SZ)--<",       "",        "",
    "DD9",             "D",       null,
    "DEPOT7",          "DEPO",    "TEBU",
    "DESIGN",          "DISEIN",  "TIZEIN",
    "DE(LMNRST)-3^",   "DE",      "TE",
    "DETTE$",          "DET",     "TET",
    "DH$",             "T",       null,
    "DIC$",            "DIZ",     "TIZ",
    "DIDR-^",          "DIT",     null,
    "DIEDR-^",         "DIT",     null,
    "DJ(AEIOU)-^",     "I",       "I",
    "DMITR-^",         "DIMIT",   "TINIT",
    "DRY9^",           "DRÜ",     null,
    "DT-",             "",        "",
    "DUIS-^",          "DÜ",      "TI",
    "DURCH^^",         "DURCH",   "TURK",
    "DVA$",            "TWA",     null,
    "DY9^",            "DÜ",      null,
    "DYS$",            "DIS",     null,
    "DS(CH)--<",       "T",       "T",
    "DST",             "ZT",      "ZT",
    "DZS(CH)--",       "T",       "T",
    "D(SßZ)",          "Z",       "Z",
    "D(AÄEIOÖRUÜY)-",  "D",       null,
    "D(ÀÁÂÃÅÈÉÊÌÍÎÙÚÛ)-", "D",    null,
    "D'H^",            "D",       "T",
    "D´H^",            "D",       "T",
    "D`H^",            "D",       "T",
    "D'S3$",           "Z",       "Z",
    "D´S3$",           "Z",       "Z",
    "D^",              "D",       null,
    "D",               "T",       "T",
    "EAULT$",          "O",       "U",
    "EAUX$",           "O",       "U",
    "EAU",             "O",       "U",
    "EAV",             "IW",      "IF",
    "EAS3$",           "EAS",     null,
    "EA(AÄEIOÖÜY)-3",  "EA",      "EA",
    "EA3$",            "EA",      "EA",
    "EA3",             "I",       "I",
    "EBENSO^$",        "EBNSO",   "EBNZU",
    "EBENSO^^",        "EBNSO ",  "EBNZU ",
    "EBEN^^",          "EBN",     "EBN",
    "EE9",             "E",       "E",
    "EGL-1",           "EK",      null,
    "EHE(IUY)--1",     "EH",      null,
    "EHUNG---1",       "E",       null,
    "EH(AÄIOÖUÜY)-1",  "EH",      null,
    "EIEI--",          "",        "",
    "EIERE^$",         "EIERE",   null,
    "EIERE$",          "EIERE",   null,
    "EIERE(NS)-$",     "EIERE",   null,
    "EIERE(AIOUY)--",  "EIER",    null,
    "EIER(AÄIOÖUÜY)-", "EIER",    null,
    "EIER<",           "EIA",     null,
    "EIGL-1",          "EIK",     null,
    "EIGH$",           "EI",      "EI",
    "EIH--",           "E",       "E",
    "EILLE$",          "EI",      "EI",
    "EIR(BCDFGKLMNQSTVWZ)-", "EIA", "EIA",
    "EIR$",            "EIA",     "EIA",
    "EITRAUBEN------", "EIT ",    "EIT ",
    "EI",              "EI",      "EI",
    "EJ$",             "EI",      "EI",
    "ELIZ^",           "ELIS",    null,
    "ELZ^",            "ELS",     null,
    "EL-^",            "E",       "E",
    "ELANG----1",      "E",       "E",
    "EL(DKL)--1",      "E",       "E",
    "EL(MNT)--1$",     "E",       "E",
    "ELYNE$",          "ELINE",   "ELINE",
    "ELYN$",           "ELIN",    "ELIN",
    "EL(AÄEÈÉÊIÌÍÎOÖUÜY)-1", "EL", "EL",
    "EL-1",            "L",       "L",
    "EM-^",            null,      "E",
    "EM(DFKMPQT)--1",  null,      "E",
    "EM(AÄEÈÉÊIÌÍÎOÖUÜY)--1", null, "E",
    "EM-1",            null,      "N",
    "ENGAG-^",         "ANGA",    "ANKA",
    "EN-^",            "E",       "E",
    "ENTUEL",          "ENTUEL",  null,
    "EN(CDGKQSTZ)--1", "E",       "E",
    "EN(AÄEÈÉÊIÌÍÎNOÖUÜY)-1", "EN", "EN",
    "EN-1",            "",        "",
    "ERH(AÄEIOÖUÜ)-^", "ERH",     "ER",
    "ER-^",            "E",       "E",
    "ERREGEND-----",   " ER",     " ER",
    "ERT1$",           "AT",      null,
    "ER(DGLKMNRQTZß)-1", "ER",    null,
    "ER(AÄEÈÉÊIÌÍÎOÖUÜY)-1", "ER", "A",
    "ER1$",            "A",       "A",
    "ER<1",            "A",       "A",
    "ETAT7",           "ETA",     "ETA",
    "ETI(AÄOÖÜU)-",    "EZI",     "EZI",
    "EUERE$",          "EUERE",   null,
    "EUERE(NS)-$",     "EUERE",   null,
    "EUERE(AIOUY)--",  "EUER",    null,
    "EUER(AÄIOÖUÜY)-", "EUER",    null,
    "EUER<",           "EUA",     null,
    "EUEU--",          "",        "",
    "EUILLE$",         "Ö",       "Ö",
    "EUR$",            "ÖR",      "ÖR",
    "EUX",             "Ö",       "Ö",
    "EUSZ$",           "EUS",     null,
    "EUTZ$",           "EUS",     null,
    "EUYS$",           "EUS",     "EUZ",
    "EUZ$",            "EUS",     null,
    "EU",              "EU",      "EU",
    "EVER--<1",        "EW",      null,
    "EV(ÄOÖUÜ)-1",     "EW",      null,
    "EYER<",           "EIA",     "EIA",
    "EY<",             "EI",      "EI",
    "FACETTE",         "FASET",   "FAZET",
    "FANS--^$",        "FE",      "FE",
    "FAN-^$",          "FE",      "FE",
    "FAULT-",          "FOL",     "FUL",
    "FEE(DL)-",        "FI",      "FI",
    "FEHLER",          "FELA",    "FELA",
    "FE(LMNRST)-3^",   "FE",      "FE",
    "FOERDERN---^",    "FÖRD",    "FÖRT",
    "FOERDERN---",     " FÖRD",   " FÖRT",
    "FOND7",           "FON",     "FUN",
    "FRAIN$",          "FRA",     "FRA",
    "FRISEU(RS)-",     "FRISÖ",   "FRIZÖ",
    "FY9^",            "FÜ",      null,
    "FÖRDERN---^",     "FÖRD",    "FÖRT",
    "FÖRDERN---",      " FÖRD",   " FÖRT",
    "GAGS^$",          "GEX",     "KEX",
    "GAG^$",           "GEK",     "KEK",
    "GD",              "KT",      "KT",
    "GEGEN^^",         "GEGN",    "KEKN",
    "GEGENGEKOM-----", "GEGN ",   "KEKN ",
    "GEGENGESET-----", "GEGN ",   "KEKN ",
    "GEGENKOMME-----", "GEGN ",   "KEKN ",
    "GEGENZUKOM---",  "GEGN ZU ", "KEKN ZU ",
    "GENDETWAS-----$", "GENT ",   "KENT ",
    "GENRE",           "IORE",    "IURE",
    "GE(LMNRST)-3^",   "GE",      "KE",
    "GER(DKT)-",       "GER",     null,
    "GETTE$",          "GET",     "KET",
    "GGF.",            "GF.",     null,
    "GG-",             "",        "",
    "GH",              "G",       null,
    "GI(AOU)-^",       "I",       "I",
    "GION-3",          "KIO",     "KIU",
    "G(CK)-",          "",        "",
    "GJ(AEIOU)-^",     "I",       "I",
    "GMBH^$",          "GMBH",    "GMBH",
    "GNAC$",           "NIAK",    "NIAK",
    "GNON$",           "NION",    "NIUN",
    "GN$",             "N",       "N",
    "GONCAL-^",        "GONZA",   "KUNZA",
    "GRY9^",           "GRÜ",     null,
    "G(SßXZ)-<",       "K",       "K",
    "GUCK-",           "KU",      "KU",
    "GUISEP-^",        "IUSE",    "IUZE",
    "GUI-^",           "G",       "K",
    "GUTAUSSEH------^", "GUT ",   "KUT ",
    "GUTGEHEND------^", "GUT ",   "KUT ",
    "GY9^",            "GÜ",      null,
    "G(AÄEILOÖRUÜY)-", "G",       null,
    "G(ÀÁÂÃÅÈÉÊÌÍÎÙÚÛ)-", "G",    null,
    "G'S$",            "X",       "X",
    "G´S$",            "X",       "X",
    "G^",              "G",       null,
    "G",               "K",       "K",
    "HA(HIUY)--1",     "H",       null,
    "HANDVOL---^",     "HANT ",   "ANT ",
    "HANNOVE-^",       "HANOF",   null,
    "HAVEN7$",         "HAFN",    null,
    "HEAD-",           "HE",      "E",
    "HELIEGEN------",  "E ",      "E ",
    "HESTEHEN------",  "E ",      "E ",
    "HE(LMNRST)-3^",   "HE",      "E",
    "HE(LMN)-1",       "E",       "E",
    "HEUR1$",          "ÖR",      "ÖR",
    "HE(HIUY)--1",     "H",       null,
    "HIH(AÄEIOÖUÜY)-1", "IH",     null,
    "HLH(AÄEIOÖUÜY)-1", "LH",     null,
    "HMH(AÄEIOÖUÜY)-1", "MH",     null,
    "HNH(AÄEIOÖUÜY)-1", "NH",     null,
    "HOBBY9^",          "HOBI",   null,
    "HOCHBEGAB-----^",  "HOCH ",  "UK ",
    "HOCHTALEN-----^",  "HOCH ",  "UK ",
    "HOCHZUFRI-----^",  "HOCH ",  "UK ",
    "HO(HIY)--1",       "H",      null,
    "HRH(AÄEIOÖUÜY)-1", "RH",     null,
    "HUH(AÄEIOÖUÜY)-1", "UH",     null,
    "HUIS^^",          "HÜS",     "IZ",
    "HUIS$",           "ÜS",      "IZ",
    "HUI--1",          "H",       null,
    "HYGIEN^",         "HÜKIEN",  null,
    "HY9^",            "HÜ",      null,
    "HY(BDGMNPST)-",   "Ü",       null,
    "H.^",             null,      "H.",
    "HÄU--1",          "H",       null,
    "H^",              "H",       "",
    "H",               "",        "",
    "ICHELL---",       "ISH",     "IZ",
    "ICHI$",           "ISHI",    "IZI",
    "IEC$",            "IZ",      "IZ",
    "IEDENSTELLE------", "IDN ",  "ITN ",
    "IEI-3",           "",        "",
    "IELL3",           "IEL",     "IEL",
    "IENNE$",          "IN",      "IN",
    "IERRE$",          "IER",     "IER",
    "IERZULAN---",     "IR ZU ",  "IR ZU ",
    "IETTE$",          "IT",      "IT",
    "IEU",             "IÖ",      "IÖ",
    "IE<4",            "I",       "I",
    "IGL-1",           "IK",      null,
    "IGHT3$",          "EIT",     "EIT",
    "IGNI(EO)-",       "INI",     "INI",
    "IGN(AEOU)-$",     "INI",     "INI",
    "IHER(DGLKRT)--1", "IHE",     null,
    "IHE(IUY)--",      "IH",      null,
    "IH(AIOÖUÜY)-",    "IH",      null,
    "IJ(AOU)-",        "I",       "I",
    "IJ$",             "I",       "I",
    "IJ<",             "EI",      "EI",
    "IKOLE$",          "IKOL",    "IKUL",
    "ILLAN(STZ)--4",   "ILIA",    "ILIA",
    "ILLAR(DT)--4",    "ILIA",    "ILIA",
    "IMSTAN----^",     "IM ",     "IN ",
    "INDELERREGE------", "INDL ", "INTL ",
    "INFRAGE-----^$",  "IN ",     "IN ",
    "INTERN(AOU)-^",   "INTAN",   "INTAN",
    "INVER-",          "INWE",    "INFE",
    "ITI(AÄIOÖUÜ)-",   "IZI",     "IZI",
    "IUSZ$",           "IUS",     null,
    "IUTZ$",           "IUS",     null,
    "IUZ$",            "IUS",     null,
    "IVER--<",         "IW",      null,
    "IVIER$",          "IWIE",    "IFIE",
    "IV(ÄOÖUÜ)-",      "IW",      null,
    "IV<3",            "IW",      null,
    "IY2",             "I",       null,
    "I(ÈÉÊ)<4",        "I",       "I",
    "JAVIE---<^",      "ZA",      "ZA",
    "JEANS^$",         "JINS",    "INZ",
    "JEANNE^$",        "IAN",     "IAN",
    "JEAN-^",          "IA",      "IA",
    "JER-^",           "IE",      "IE",
    "JE(LMNST)-",      "IE",      "IE",
    "JI^",             "JI",      null,
    "JOR(GK)^$",       "IÖRK",    "IÖRK",
    "J",               "I",       "I",
    "KC(ÄEIJ)-",       "X",       "X",
    "KD",              "KT",      null,
    "KE(LMNRST)-3^",   "KE",      "KE",
    "KG(AÄEILOÖRUÜY)-", "K",      null,
    "KH<^",            "K",       "K",
    "KIC$",            "KIZ",     "KIZ",
    "KLE(LMNRST)-3^",  "KLE",     "KLE",
    "KOTELE-^",        "KOTL",    "KUTL",
    "KREAT-^",         "KREA",    "KREA",
    "KRÜS(TZ)--^",     "KRI",     null,
    "KRYS(TZ)--^",     "KRI",     null,
    "KRY9^",           "KRÜ",     null,
    "KSCH---",         "K",       "K",
    "KSH--",           "K",       "K",
    "K(SßXZ)7",        "X",       "X",    /**** implies "KST" -> "XT" ****/
    "KT'S$",           "X",       "X",
    "KTI(AIOU)-3",     "XI",      "XI",
    "KT(SßXZ)",        "X",       "X",
    "KY9^",            "KÜ",      null,
    "K'S$",            "X",       "X",
    "K´S$",            "X",       "X",
    "LANGES$",         " LANGES", " LANKEZ",
    "LANGE$",          " LANGE",  " LANKE",
    "LANG$",           " LANK",   " LANK",
    "LARVE-",          "LARF",    "LARF",
    "LD(SßZ)$",        "LS",      "LZ",
    "LD'S$",           "LS",      "LZ",
    "LD´S$",           "LS",      "LZ",
    "LEAND-^",         "LEAN",    "LEAN",
    "LEERSTEHE-----^", "LER ",    "LER ",
    "LEICHBLEIB-----", "LEICH ",  "LEIK ",
    "LEICHLAUTE-----", "LEICH ",  "LEIK ",
    "LEIDERREGE------", "LEIT ",  "LEIT ",
    "LEIDGEPR----^",   "LEIT ",   "LEIT ",
    "LEINSTEHE-----",  "LEIN ",   "LEIN ",
    "LEL-",            "LE",      "LE",
    "LE(MNRST)-3^",    "LE",      "LE",
    "LETTE$",          "LET",     "LET",
    "LFGNAG-",         "LFGAN",   "LFKAN",
    "LICHERWEIS----",  "LICHA ", "LIKA ",
    "LIC$",            "LIZ",     "LIZ",
    "LIVE^$",          "LEIF",    "LEIF",
    "LT(SßZ)$",        "LS",      "LZ",
    "LT'S$",           "LS",      "LZ",
    "LT´S$",           "LS",      "LZ",
    "LUI(GS)--",       "LU",      "LU",
    "LV(AIO)-",        "LW",      null,
    "LY9^",            "LÜ",      null,
    "LSTS$",           "LS",      "LZ",
    "LZ(BDFGKLMNPQRSTVWX)-", "LS", null,
    "L(SßZ)$",         "LS",      null,
    "MAIR-<",          "MEI",     "NEI",
    "MANAG-",          "MENE",    "NENE",
    "MANUEL",          "MANUEL",  null,
    "MASSEU(RS)-",     "MASÖ",    "NAZÖ",
    "MATCH",           "MESH",    "NEZ",
    "MAURICE",         "MORIS",   "NURIZ",
    "MBH^$",           "MBH",     "MBH",
    "MB(ßZ)$",         "MS",      null,
    "MB(SßTZ)-",       "M",       "N",
    "MCG9^",           "MAK",     "NAK",
    "MC9^",            "MAK",     "NAK",
    "MEMOIR-^",        "MEMOA",   "NENUA",
    "MERHAVEN$",       "MAHAFN",  null,
    "ME(LMNRST)-3^",   "ME",      "NE",
    "MEN(STZ)--3",     "ME",      null,
    "MEN$",            "MEN",     null,
    "MIGUEL-",         "MIGE",    "NIKE",
    "MIKE^$",          "MEIK",    "NEIK",
    "MITHILFE----^$",  "MIT H",   "NIT ",
    "MN$",             "M",       null,
    "MN",              "N",       "N",
    "MPJUTE-",         "MPUT",    "NBUT",
    "MP(ßZ)$",         "MS",      null,
    "MP(SßTZ)-",       "M",       "N",
    "MP(BDJLMNPQVW)-", "MB",      "NB",
    "MY9^",            "MÜ",      null,
    "M(ßZ)$",          "MS",      null,
    "M´G7^",           "MAK",     "NAK",
    "M'G7^",           "MAK",     "NAK",
    "M´^",             "MAK",     "NAK",
    "M'^",             "MAK",     "NAK",
    "M",               null,      "N",
    "NACH^^",          "NACH",    "NAK",
    "NADINE",          "NADIN",   "NATIN",
    "NAIV--",          "NA",      "NA",
    "NAISE$",          "NESE",    "NEZE",
    "NAUGENOMM------", "NAU ",    "NAU ",
    "NAUSOGUT$",     "NAUSO GUT", "NAUZU KUT",
    "NCH$",            "NSH",     "NZ",
    "NCOISE$",         "SOA",     "ZUA",
    "NCOIS$",          "SOA",     "ZUA",
    "NDAR$",           "NDA",     "NTA",
    "NDERINGEN------", "NDE ",    "NTE ",
    "NDRO(CDKTZ)-",    "NTRO",    null,
    "ND(BFGJLMNPQVW)-", "NT",     null,
    "ND(SßZ)$",        "NS",      "NZ",
    "ND'S$",           "NS",      "NZ",
    "ND´S$",           "NS",      "NZ",
    "NEBEN^^",         "NEBN",    "NEBN",
    "NENGELERN------", "NEN ",    "NEN ",
    "NENLERN(ET)---",  "NEN LE",  "NEN LE",
    "NENZULERNE---", "NEN ZU LE", "NEN ZU LE",
    "NE(LMNRST)-3^",   "NE",      "NE",
    "NEN-3",           "NE",      "NE",
    "NETTE$",          "NET",     "NET",
    "NGU^^",           "NU",      "NU",
    "NG(BDFJLMNPQRTVW)-", "NK",   "NK",
    "NH(AUO)-$",       "NI",      "NI",
    "NICHTSAHNEN-----", "NIX ",   "NIX ",
    "NICHTSSAGE----",  "NIX ",    "NIX ",
    "NICHTS^^",        "NIX",     "NIX",
    "NICHT^^",         "NICHT",   "NIKT",
    "NINE$",           "NIN",     "NIN",
    "NON^^",           "NON",     "NUN",
    "NOTLEIDE-----^",  "NOT ",   "NUT ",
    "NOT^^",           "NOT",     "NUT",
    "NTI(AIOU)-3",     "NZI",     "NZI",
    "NTIEL--3",        "NZI",     "NZI",
    "NT(SßZ)$",        "NS",      "NZ",
    "NT'S$",           "NS",      "NZ",
    "NT´S$",           "NS",      "NZ",
    "NYLON",           "NEILON",  "NEILUN",
    "NY9^",            "NÜ",      null,
    "NSTZUNEH---",     "NST ZU ", "NZT ZU ",
    "NSZ-",            "NS",      null,
    "NSTS$",           "NS",      "NZ",
    "NZ(BDFGKLMNPQRSTVWX)-", "NS", null,
    "N(SßZ)$",         "NS",      null,
    "OBERE-",          "OBER",    null,
    "OBER^^",          "OBA",     "UBA",
    "OEU2",            "Ö",       "Ö",
    "OE<2",            "Ö",       "Ö",
    "OGL-",            "OK",      null,
    "OGNIE-",          "ONI",     "UNI",
    "OGN(AEOU)-$",     "ONI",     "UNI",
    "OH(AIOÖUÜY)-",    "OH",      null,
    "OIE$",            "Ö",       "Ö",
    "OIRE$",           "OA",      "UA",
    "OIR$",            "OA",      "UA",
    "OIX",             "OA",      "UA",
    "OI<3",            "EU",      "EU",
    "OKAY^$",          "OKE",     "UKE",
    "OLYN$",           "OLIN",    "ULIN",
    "OO(DLMZ)-",       "U",       null,
    "OO$",             "U",       null,
    "OO-",             "",        "",
    "ORGINAL-----",    "ORI",     "URI",
    "OTI(AÄOÖUÜ)-",    "OZI",     "UZI",
    "OUI^",            "WI",      "FI",
    "OUILLE$",         "ULIE",    "ULIE",
    "OU(DT)-^",        "AU",      "AU",
    "OUSE$",           "AUS",     "AUZ",
    "OUT-",            "AU",      "AU",
    "OU",              "U",       "U",
    "O(FV)$",          "AU",      "AU",  /****  due to "OW$" -> "AU"  ****/
    "OVER--<",         "OW",      null,
    "OV(AOU)-",        "OW",      null,
    "OW$",             "AU",      "AU",
    "OWS$",            "OS",      "UZ",
    "OJ(AÄEIOÖUÜ)--",  "O",       "U",
    "OYER",            "OIA",     null,
    "OY(AÄEIOÖUÜ)--",  "O",       "U",
    "O(JY)<",          "EU",      "EU",
    "OZ$",             "OS",      null,
    "O´^",             "O",       "U",
    "O'^",             "O",       "U",
    "O",               null,      "U",
    "PATIEN--^",       "PAZI",    "PAZI",
    "PENSIO-^",        "PANSI",   "PANZI",
    "PE(LMNRST)-3^",   "PE",      "PE",
    "PFER-^",          "FE",      "FE",
    "P(FH)<",          "F",       "F",
    "PIC^$",           "PIK",     "PIK",
    "PIC$",            "PIZ",     "PIZ",
    "PIPELINE",       "PEIBLEIN", "PEIBLEIN",
    "POLYP-",          "POLÜ",    null,
    "POLY^^",          "POLI",    "PULI",
    "PORTRAIT7",       "PORTRE",  "PURTRE",
    "POWER7",          "PAUA",    "PAUA",
    "PP(FH)--<",       "B",       "B",
    "PP-",             "",        "",
    "PRODUZ-^",        "PRODU",   "BRUTU",
    "PRODUZI--",       " PRODU",  " BRUTU",
    "PRIX^$",          "PRI",     "PRI",
    "PS-^^",           "P",       null,
    "P(SßZ)^",         null,      "Z",
    "P(SßZ)$",         "BS",      null,
    "PT-^",            "",        "",
    "PTI(AÄOÖUÜ)-3",   "BZI",     "BZI",
    "PY9^",            "PÜ",      null,
    "P(AÄEIOÖRUÜY)-",  "P",       "P",
    "P(ÀÁÂÃÅÈÉÊÌÍÎÙÚÛ)-", "P",    null,
    "P.^",             null,      "P.",
    "P^",              "P",       null,
    "P",               "B",       "B",
    "QI-",             "Z",       "Z",
    "QUARANT--",       "KARA",    "KARA",
    "QUE(LMNRST)-3",   "KWE",     "KFE",
    "QUE$",            "K",       "K",
    "QUI(NS)$",        "KI",      "KI",
    "QUIZ7",           "KWIS",    null,
    "Q(UV)7",          "KW",      "KF",
    "Q<",              "K",       "K",
    "RADFAHR----",     "RAT ",    "RAT ",
    "RAEFTEZEHRE-----", "REFTE ", "REFTE ",
    "RCH",             "RCH",     "RK",
    "REA(DU)---3^",    "R",       null,
    "REBSERZEUG------", "REBS ",  "REBZ ",
    "RECHERCH^",       "RESHASH", "REZAZ",
    "RECYCL--",        "RIZEI",  "RIZEI",
    "RE(ALST)-3^",     "RE",      null,
    "REE$",            "RI",      "RI",
    "RER$",            "RA",      "RA",
    "RE(MNR)-4",       "RE",      "RE",
    "RETTE$",          "RET",     "RET",
    "REUZ$",           "REUZ",    null,
    "REW$",            "RU",      "RU",
    "RH<^",            "R",       "R",
    "RJA(MN)--",       "RI",      "RI",
    "ROWD-^",          "RAU",     "RAU",
    "RTEMONNAIE-",     "RTMON",   "RTNUN",
    "RTI(AÄOÖUÜ)-3",   "RZI",     "RZI",
    "RTIEL--3",        "RZI",     "RZI",
    "RV(AEOU)-3",      "RW",      null,
    "RY(KN)-$",        "RI",      "RI",
    "RY9^",            "RÜ",      null,
    "RÄFTEZEHRE-----", "REFTE ",  "REFTE ",
    "SAISO-^",         "SES",     "ZEZ",
    "SAFE^$",          "SEIF",    "ZEIF",
    "SAUCE-^",         "SOS",     "ZUZ",
    "SCHLAGGEBEN-----<", "SHLAK ", "ZLAK ",
    "SCHSCH---7",      "",        "",
    "SCHTSCH",         "SH",      "Z",
    "SC(HZ)<",         "SH",      "Z",
    "SC",              "SK",      "ZK",
    "SELBSTST--7^^",   "SELB",    "ZELB",
    "SELBST7^^",       "SELBST",  "ZELBZT",
    "SERVICE7^",       "SÖRWIS",  "ZÖRFIZ",
    "SERVI-^",         "SERW",    null,
    "SE(LMNRST)-3^",   "SE",      "ZE",
    "SETTE$",          "SET",     "ZET",
    "SHP-^",           "S",       "Z",
    "SHST",            "SHT",     "ZT",
    "SHTSH",           "SH",      "Z",
    "SHT",             "ST",      "Z",
    "SHY9^",           "SHÜ",     null,
    "SH^^",            "SH",      null,
    "SH3",             "SH",      "Z",
    "SICHERGEGAN-----^", "SICHA ", "ZIKA ",
    "SICHERGEHE----^", "SICHA ",  "ZIKA ",
    "SICHERGESTEL------^", "SICHA ", "ZIKA ",
    "SICHERSTELL-----^", "SICHA ", "ZIKA ",
    "SICHERZU(GS)--^", "SICHA ZU ", "ZIKA ZU ",
    "SIEGLI-^",        "SIKL",    "ZIKL",
    "SIGLI-^",         "SIKL",    "ZIKL",
    "SIGHT",           "SEIT",    "ZEIT",
    "SIGN",            "SEIN",    "ZEIN",
    "SKI(NPZ)-",       "SKI",     "ZKI",
    "SKI<^",           "SHI",     "ZI",
    "SODASS^$",        "SO DAS",  "ZU TAZ",
    "SODAß^$",         "SO DAS",  "ZU TAZ",
    "SOGENAN--^",      "SO GEN",  "ZU KEN",
    "SOUND-",          "SAUN",    "ZAUN",
    "STAATS^^",        "STAZ",    "ZTAZ",
    "STADT^^",         "STAT",    "ZTAT",
    "STANDE$",         " STANDE", " ZTANTE",
    "START^^",         "START",   "ZTART",
    "STAURANT7",       "STORAN",  "ZTURAN",
    "STEAK-",          "STE",     "ZTE",
    "STEPHEN-^$",      "STEW",    null,
    "STERN",           "STERN",   null,
    "STRAF^^",         "STRAF",   "ZTRAF",
    "ST'S$",           "Z",       "Z",
    "ST´S$",           "Z",       "Z",
    "STST--",          "",        "",
    "STS(ACEÈÉÊHIÌÍÎOUÄÜÖ)--", "ST", "ZT",
    "ST(SZ)",          "Z",       "Z",
    "SPAREN---^",      "SPA",     "ZPA",
    "SPAREND----",     " SPA",    " ZPA",
    "S(PTW)-^^",       "S",       null,
    "SP",              "SP",      null,
    "STYN(AE)-$",      "STIN",    "ZTIN",
    "ST",              "ST",      "ZT",
    "SUITE<",          "SIUT",    "ZIUT",
    "SUKE--$",         "S",       "Z",
    "SURF(EI)-",       "SÖRF",    "ZÖRF",
    "SV(AEÈÉÊIÌÍÎOU)-<^", "SW",   null,
    "SYB(IY)--^",      "SIB",     null,
    "SYL(KVW)--^",     "SI",      null,
    "SY9^",            "SÜ",      null,
    "SZE(NPT)-^",      "ZE",      "ZE",
    "SZI(ELN)-^",      "ZI",      "ZI",
    "SZCZ<",           "SH",      "Z",
    "SZT<",            "ST",      "ZT",
    "SZ<3",            "SH",      "Z",
    "SÜL(KVW)--^",     "SI",      null,
    "S",               null,      "Z",
    "TCH",             "SH",      "Z",
    "TD(AÄEIOÖRUÜY)-", "T",       null,
    "TD(ÀÁÂÃÅÈÉÊËÌÍÎÏÒÓÔÕØÙÚÛÝŸ)-", "T", null,
    "TEAT-^",          "TEA",     "TEA",
    "TERRAI7^",        "TERA",    "TERA",
    "TE(LMNRST)-3^",   "TE",      "TE",
    "TH<",             "T",       "T",
    "TICHT-",          "TIK",     "TIK",
    "TICH$",           "TIK",     "TIK",
    "TIC$",            "TIZ",     "TIZ",
    "TIGGESTELL-------", "TIK ",  "TIK ",
    "TIGSTELL-----",   "TIK ",    "TIK ",
    "TOAS-^",          "TO",      "TU",
    "TOILET-",         "TOLE",    "TULE",
    "TOIN-",           "TOA",     "TUA",
    "TRAECHTI-^",      "TRECHT",  "TREKT",
    "TRAECHTIG--",     " TRECHT", " TREKT",
    "TRAINI-",         "TREN",    "TREN",
    "TRÄCHTI-^",       "TRECHT",  "TREKT",
    "TRÄCHTIG--",      " TRECHT", " TREKT",
    "TSCH",            "SH",      "Z",
    "TSH",             "SH",      "Z",
    "TST",             "ZT",      "ZT",
    "T(Sß)",           "Z",       "Z",
    "TT(SZ)--<",       "",        "",
    "TT9",             "T",       "T",
    "TV^$",            "TV",      "TV",
    "TX(AEIOU)-3",     "SH",      "Z",
    "TY9^",            "TÜ",      null,
    "TZ-",             "",        "",
    "T'S3$",           "Z",       "Z",
    "T´S3$",           "Z",       "Z",
    "UEBEL(GNRW)-^^",  "ÜBL ",    "IBL ",
    "UEBER^^",         "ÜBA",     "IBA",
    "UE2",             "Ü",       "I",
    "UGL-",            "UK",      null,
    "UH(AOÖUÜY)-",     "UH",      null,
    "UIE$",            "Ü",       "I",
    "UM^^",            "UM",      "UN",
    "UNTERE--3",       "UNTE",    "UNTE",
    "UNTER^^",         "UNTA",    "UNTA",
    "UNVER^^",         "UNFA",    "UNFA",
    "UN^^",            "UN",      "UN",
    "UTI(AÄOÖUÜ)-",    "UZI",     "UZI",
    "UVE-4",           "UW",      null,
    "UY2",             "UI",      null,
    "UZZ",             "AS",      "AZ",
    "VACL-^",          "WAZ",     "FAZ",
    "VAC$",            "WAZ",     "FAZ",
    "VAN DEN ^",       "FANDN",   "FANTN",
    "VANES-^",         "WANE",    null,
    "VATRO-",          "WATR",    null,
    "VA(DHJNT)--^",    "F",       null,
    "VEDD-^",          "FE",      "FE",
    "VE(BEHIU)--^",    "F",       null,
    "VEL(BDLMNT)-^",   "FEL",     null,
    "VENTZ-^",         "FEN",     null,
    "VEN(NRSZ)-^",     "FEN",     null,
    "VER(AB)-^$",      "WER",     null,
    "VERBAL^$",        "WERBAL",  null,
    "VERBAL(EINS)-^",  "WERBAL",  null,
    "VERTEBR--",       "WERTE",   null,
    "VEREIN-----",     "F",       null,
    "VEREN(AEIOU)-^",  "WEREN",   null,
    "VERIFI",          "WERIFI",  null,
    "VERON(AEIOU)-^",  "WERON",   null,
    "VERSEN^",         "FERSN",   "FAZN",
    "VERSIERT--^",     "WERSI",   null,
    "VERSIO--^",       "WERS",    null,
    "VERSUS",          "WERSUS",  null,
    "VERTI(GK)-",      "WERTI",   null,
    "VER^^",           "FER",     "FA",
    "VERSPRECHE-------", " FER",  " FA",
    "VER$",            "WA",      null,
    "VER",             "FA",      "FA",
    "VET(HT)-^",       "FET",     "FET",
    "VETTE$",          "WET",     "FET",
    "VE^",             "WE",      null,
    "VIC$",            "WIZ",     "FIZ",
    "VIELSAGE----",    "FIL ",    "FIL ",
    "VIEL",            "FIL",     "FIL",
    "VIEW",            "WIU",     "FIU",
    "VILL(AE)-",       "WIL",     null,
    "VIS(ACEIKUVWZ)-<^", "WIS",   null,
    "VI(ELS)--^",      "F",       null,
    "VILLON--",        "WILI",    "FILI",
    "VIZE^^",          "FIZE",    "FIZE",
    "VLIE--^",         "FL",      null,
    "VL(AEIOU)--",     "W",       null,
    "VOKA-^",          "WOK",     null,
    "VOL(ATUVW)--^",   "WO",      null,
    "VOR^^",           "FOR",     "FUR",
    "VR(AEIOU)--",     "W",       null,
    "VV9",             "W",       null,
    "VY9^",            "WÜ",      "FI",
    "V(ÜY)-",          "W",       null,
    "V(ÀÁÂÃÅÈÉÊÌÍÎÙÚÛ)-",  "W",   null,
    "V(AEIJLRU)-<",    "W",       null,
    "V.^",             "V.",      null,
    "V<",              "F",       "F",
    "WEITERENTWI-----^", "WEITA ", "FEITA ",
    "WEITREICH-----^", "WEIT ",   "FEIT ",
    "WEITVER^",       "WEIT FER", "FEIT FA",
    "WE(LMNRST)-3^",   "WE",      "FE",
    "WER(DST)-",       "WER",     null,
    "WIC$",            "WIZ",     "FIZ",
    "WIEDERU--",       "WIDE",    "FITE",
    "WIEDER^$",        "WIDA",    "FITA",
    "WIEDER^^",        "WIDA ",   "FITA ",
    "WIEVIEL",         "WI FIL",  "FI FIL",
    "WISUEL",          "WISUEL",  null,
    "WR-^",            "W",       null,
    "WY9^",            "WÜ",      "FI",
    "W(BDFGJKLMNPQRSTZ)-", "F",   null,
    "W$",              "F",       null,
    "W",               null,      "F",
    "X<^",             "Z",       "Z",
    "XHAVEN$",         "XAFN",    null,
    "X(CSZ)",          "X",       "X",
    "XTS(CH)--",       "XT",      "XT",
    "XT(SZ)",          "Z",       "Z",
    "YE(LMNRST)-3^",   "IE",      "IE",
    "YE-3",            "I",       "I",
    "YOR(GK)^$",       "IÖRK",    "IÖRK",
    "Y(AOU)-<7",       "I",       "I",
    "Y(BKLMNPRSTX)-1", "Ü",       null,
    "YVES^$",          "IF",      "IF",
    "YVONNE^$",        "IWON",    "IFUN",
    "Y.^",             "Y.",      null,
    "Y",               "I",       "I",
    "ZC(AOU)-",        "SK",      "ZK",
    "ZE(LMNRST)-3^",   "ZE",      "ZE",
    "ZIEJ$",           "ZI",      "ZI",
    "ZIGERJA(HR)-3",   "ZIGA IA", "ZIKA IA",
    "ZL(AEIOU)-",      "SL",      null,
    "ZS(CHT)--",       "",        "",
    "ZS",              "SH",      "Z",
    "ZUERST",          "ZUERST",  "ZUERST",
    "ZUGRUNDE^$",    "ZU GRUNDE", "ZU KRUNTE",
    "ZUGRUNDE",     "ZU GRUNDE ", "ZU KRUNTE ",
    "ZUGUNSTEN",     "ZU GUNSTN", "ZU KUNZTN",
    "ZUHAUSE-",        "ZU HAUS", "ZU AUZ",
    "ZULASTEN^$",     "ZU LASTN", "ZU LAZTN",
    "ZURUECK^^",       "ZURÜK",   "ZURIK",
    "ZURZEIT",        "ZUR ZEIT", "ZUR ZEIT",
    "ZURÜCK^^",        "ZURÜK",   "ZURIK",
    "ZUSTANDE",      "ZU STANDE", "ZU ZTANTE",
    "ZUTAGE",          "ZU TAGE", "ZU TAKE",
    "ZUVER^^",         "ZUFA",    "ZUFA",
    "ZUVIEL",          "ZU FIL",  "ZU FIL",
    "ZUWENIG",        "ZU WENIK", "ZU FENIK",
    "ZY9^",            "ZÜ",      null,
    "ZYK3$",           "ZIK",     null,
    "Z(VW)7^",         "SW",      null,
    PHONET_END,      PHONET_END,  PHONET_END
  };

#endif    /****  PHONET_GERMAN  ****/




static int set_phonet_language (int rule_set)
/****  result:  0 :  success             ****/
/****          -1 :  language not found  ****/
{
 switch (rule_set)
   {
    #ifdef PHONET_NO_LANGUAGE
         case PHONET_NO_LANGUAGE :
                   phonet_language = phonet_language_no_language;
                   phonet_init   = & phonet_init_no_language;
                   phonet_hash   = phonet_hash_no_language;
                   phonet_hash_1 = & phonet_hash_no_language_1;
                   phonet_hash_2 = & phonet_hash_no_language_2;
                   phonet_rules  = phonet_rules_no_language;
             return (0);
    #endif

    #ifdef PHONET_ENGLISH
         case PHONET_ENGLISH :
                   phonet_language = phonet_language_english;
                   phonet_init   = & phonet_init_english;
                   phonet_hash   = phonet_hash_english;
                   phonet_hash_1 = & phonet_hash_english_1;
                   phonet_hash_2 = & phonet_hash_english_2;
                   phonet_rules  = phonet_rules_english;
             return (0);
    #endif

    #ifdef PHONET_GERMAN
         case PHONET_GERMAN  :
                   phonet_language = phonet_language_german;
                   phonet_init   = & phonet_init_german;
                   phonet_hash   = phonet_hash_german;
                   phonet_hash_1 = & phonet_hash_german_1;
                   phonet_hash_2 = & phonet_hash_german_2;
                   phonet_rules  = phonet_rules_german;
             return (0);
    #endif
   }
 return (-1);
}

#endif     /****  _PHONET_H_  ****/


/************************************************************/
/****  end of file "phonet.h"  ******************************/
/************************************************************/
