## extract tags
tag <- function(x) attr(x, "Rd_tag")

## replace tags
untag <- function(x) {
  if (is.null(x)) return(NULL)
  attr(x, "Rd_tag") <- "TEXT"
  x
}

## construct strings from rd
reconstruct <- function(rd) {
  if (is.null(rd)) return()

  if (is.list(rd)) {
	if (length(tag(rd)) && tag(rd) %in% c('\\item', '\\tabular', '\\eqn', '\\deqn', '\\link')){
		if (tag(rd) == '\\link')
			return(paste('\\link', sprintf('[%s]', attr(rd, 'Rd_option')), '{', rd, '}', sep = ""))
		if (length(rd) == 2)
			return(paste(tag(rd), '{', rd[[1]], '}{',
				paste(sapply(rd[[2]], reconstruct), collapse = ""),
				'}', sep = "", collapse = "")) else if (length(rd) == 0) return(tag(rd))
	}
	special <- tag(rd) == toupper(tag(rd))
	singles <- tag(rd) %in% c('\\tab', '\\cr')
	prefix <- ifelse(special, "",
		paste(tag(rd), ifelse(singles, "", "{"), sep = ""))
	suffix <- ifelse(special, "", ifelse(singles, "", "}"))
	paste(prefix, paste(sapply(rd, reconstruct), collapse = ""), suffix,
		 sep = "")
  } else {
    rd
  }
}

## wrap strings with comment prefix
comment_line <- function(x, exdent = 0) {
  if (missing(x)) return(comment_prefix())

  strwrap(x, width = 80, exdent = exdent, prefix = comment_prefix())
}

## add comments
comment_tag <- function(tag, value) {
  if (is.null(value) || value == "" || length(value) == 0) return()

  comment_line(paste(tag, value), exdent = 2)
}

## access the comment prefix
comment_prefix <- function() {
	if (is.null(getOption("roxygen.comment")))
				"#' " else getOption("roxygen.comment")
}


##' Roxygenize a package, clean up and build/check the package
##'
##' After the source package is roxygenized, this function will build
##' the package. Optionally it also installs or checks the package,
##' reformats the code in the examples section. Note \code{\link{rab}}
##' is an alias of \code{\link{roxygen_and_build}}.
##'
##' @param pkg the root directory of the source package
##' @param roxygen.dir the directory for the roxygenized package (by
##' default it is the same as \code{pkg})
##' @param build whether to build the package
##' @param install whether to install the package
##' @param check whether to check the package
##' @param check.opts options to check the package
##' (e.g. \code{"--no-examples"})
##' @param remove.check whether to remove the directory generated by
##' \command{R CMD check}
##' @param reformat whether to reformat the examples code; see
##' \code{\link{reformat_code}}
##' @param ... other arguments passed to
##' \code{\link[roxygen2]{roxygenize}}
##' @return \code{NULL}
##' @author Yihui Xie <\url{http://yihui.name}>
##' @rdname roxygen_and_build
##' @export
##' @examples \dontrun{
##' roxygen_and_build("Rd2roxygen", install = TRUE)
##' ## or simply
##' rab('Rd2roxygen', install = TRUE)
##' }
roxygen_and_build = function(pkg, roxygen.dir = pkg, build = TRUE, install = FALSE,
    check = FALSE, check.opts = "", remove.check = TRUE, reformat = TRUE, ...) {
    roxygenize(pkg, roxygen.dir, ...)
    if (normalizePath(pkg) != normalizePath(roxygen.dir))
        unlink(sprintf("%s/.git", roxygen.dir), recursive = TRUE)
    rd.list = list.files(file.path(roxygen.dir, "man"), ".*\\.Rd$", all.files = TRUE, full.names = TRUE)
    if (reformat) {
        for (f in rd.list) {
            if (isTRUE(reformat_code(f))) {
                x = readLines(f)
                x = gsub('\\\\dontrun', '\\dontrun', x, fixed = TRUE)
                writeLines(x, con = f)
            }
        }
    }
    if (build) system(sprintf("R CMD build %s ", roxygen.dir))
    if (install) system(sprintf("R CMD INSTALL %s ", roxygen.dir))
    if (check) {
        if ((system(sprintf("R CMD check %s %s", roxygen.dir, check.opts)) == 0) &&
            remove.check) unlink(sprintf('%s.Rcheck', roxygen.dir), TRUE)
    }
    invisible(NULL)
}

##' @rdname roxygen_and_build
##' @export
rab = roxygen_and_build


##' Format the code in the examples section
##'
##' The function \code{\link[formatR]{tidy.source}} in the
##' \pkg{formatR} package is used to polish the Rd files generated by
##' \pkg{roxygen2} in the examples section.
##'
##' If the macro \code{"\\dontrun"} is detected in the Rd file, this
##' function will use an unstable way to reformat the examples
##' section, which might destroy the original Rd (e.g. the subsection
##' macro can be mangled); otherwise it will use ordinary
##' text-processing techniques to deal with these sections and
##' generally will not affect other sections. However, neither way is
##' absolutely safe. Whenever you run into troubles, just try to turn
##' off reformatting.
##'
##' @param path the path of the Rd file
##' @param ... other arguments passed to \code{tidy.source}
##' @return \code{TRUE} if the macro \code{"\\dontrun"} is detected in
##' the Rd; \code{FALSE} if not detected or there is no examples
##' section or it fails to reformat the code; as a side effect, the
##' original Rd file will be updated
##' @export
##' @author Yihui Xie <\url{http://yihui.name}>
##' @seealso \code{\link[formatR]{tidy.source}}
##' @note If the examples code is not syntactically correct, it will
##' not be reformatted and a message will be printed on screen. One
##' possible situation is the percent symbol \code{\%}, which should
##' be escaped even in the examples code (cf Writing R Extensions),
##' and this can make the code syntactically incorrect, e.g. \code{a
##' \%in\% b} should be \code{a \\\%in\\\% b} but the latter is not
##' valid R code. Anyway, this function will try to unescape the
##' percent symbols before reformating the code, then escape them.
##' @examples
##' rd.file = system.file('examples', 'reformat_code_demo.Rd', package = 'Rd2roxygen')
##' file.copy(rd.file, tempdir())
##' fmt.file = file.path(tempdir(), 'reformat_code_demo.Rd')
##'
##' file.show(fmt.file)  ## show the raw Rd
##'
##' reformat_code(fmt.file)
##' file.show(fmt.file)  ## the formatted Rd
reformat_code = function(path, ...) {
    rd = readLines(path)
    idx0 = grep('^\\\\examples\\{', rd)
    if (!length(idx0)) return(invisible(FALSE))
    ## easy to deal with when there are no \\dontrun's
    if (!any(grepl('\\\\dontrun', rd[idx0:length(rd)]))) {
        ## remove trailing spaces
        while (tail(rd, 1) == '') {
            rd = rd[-length(rd)]
        }
        ## these tags might appear after \examples (are there other possibilities??)
        idx1 = grep('^\\\\(author|seealso|keyword)\\{', rd)
        if (length(idx1) && any(idx1 > idx0))
            idx1 = min(idx1[idx1 > idx0]) - 1 else idx1 = length(rd)
        tmp = rd[idx0:idx1]
        tmp[1] = sub('^\\\\examples\\{', '', tmp[1])
        nn = length(tmp)
        tmp[nn] = sub('\\}$', '', tmp[nn])
        tmp = gsub('\\%', '%', tmp, fixed = TRUE) # will escape % later
        txt =
            try(tidy.source(text = tmp, output = FALSE, keep.blank.line = TRUE,
                            ...)$text.tidy, silent = TRUE)
        if (!inherits(txt, 'try-error')) {
            txt = gsub("(^|[^\\])%", "\\1\\\\%", txt)
            txt[1] = paste('\\examples{', txt[1], sep = '')
            nn0 = length(txt)
            txt[nn0] = paste(txt[nn0], '}', sep = '')
            rd[idx0] = paste(txt, collapse = '\n')
            if (idx1 > idx0)
                rd = rd[-((idx0 + 1):idx1)]
        } else {
            message('(!) failed to reformat examples code in ', path)
            message(paste('   ', txt, collapse = '\n'))
        }
        writeLines(rd, path)
        flush.console()
        return(invisible(FALSE))
    }
    rd = tools::parse_Rd(path)
    idx = which(sapply(rd, tag) == '\\examples')
    txt = rd[idx]
    class(txt) = 'Rd'
    txt = as.character(txt)
    txt = txt[-c(1, 2, length(txt))]
    txt = gsub('^[[:space:]]+', '', txt)
    txt = sub('^\\\\dontrun', 'tag_name_dontrun = function() ', txt)
    txt[txt == ''] = '\n'
    txt = gsub('\\%', '%', txt, fixed = TRUE)
    txt =
        try(tidy.source(text = txt, output = FALSE, keep.blank.line = TRUE, ...),
            silent = TRUE)
    if (!inherits(txt, 'try-error')) {
        txt = txt$text.tidy
        txt = gsub("(^|[^\\])%", "\\1\\\\%", txt)
        txt = gsub('tag_name_dontrun = function() {', '\\dontrun{', txt, fixed = TRUE)
        txt = unlist(strsplit(txt, '\n'))
        ## remove the four spaces introduced by disguising \\dontrun as a function
        if (length(idx0 <- grep('\\dontrun{', txt, fixed = TRUE))) {
            for (i in idx0) {
                j = i + 1
                while (txt[j] != '}') {
                    txt[j] = sub('^    ', '', txt[j])
                    j = j + 1
                }
            }
        }
        txt = paste(txt, rep(c('\n', ''), c(length(txt) - 1, 1)), sep = '', collapse = '')
        rd[[idx]] = structure(list(structure(txt, Rd_tag = 'TEXT')), Rd_tag = '\\examples')
        class(rd) = 'Rd'
        cat(as.character(rd, deparse = TRUE), file = path, sep = '')
        message('(!) reformatting might destroy your Rd without warnings,\n',
                '    because the \\dontrun macro is present in ', path)
    } else {
        message('(!) failed to reformat examples code in ', path)
        flush.console()
        return(invisible(FALSE))
    }
    invisible(TRUE)
}
