% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calc_ete.R
\name{calc_ete}
\alias{calc_ete}
\title{Calculates the Effective Transfer Entropy for two time series}
\usage{
calc_ete(x, y, lx = 1, ly = 1, q = 0.1, entropy = "Shannon",
  shuffles = 100, type = "quantiles", quantiles = c(5, 95),
  bins = NULL, limits = NULL, burn = 50, seed = NULL,
  na.rm = TRUE)
}
\arguments{
\item{x}{a vector of numeric values, ordered by time.
Also allowed are \code{\link[xts]{xts}}, \code{\link[zoo]{zoo}},
or \code{\link[stats]{ts}} objects.}

\item{y}{a vector of numeric values, ordered by time.
Also allowed are \code{\link[xts]{xts}}, \code{\link[zoo]{zoo}},
or \code{\link[stats]{ts}} objects.}

\item{lx}{Markov order of x, i.e. the number of lagged values affecting the
current value of x. Default is \code{lx = 1}.}

\item{ly}{Markov order of y, i.e. the number of lagged values affecting the
current value of y. Default is \code{ly = 1}.}

\item{q}{a weighting parameter used to estimate Renyi transfer entropy,
parameter is between 0 and 1. For \code{q = 1}, Renyi transfer
entropy converges to Shannon transfer entropy. Default is
\code{q = 0.1}.}

\item{entropy}{specifies the transfer entropy measure that is estimated,
either 'Shannon' or 'Renyi'. The first character can be used
to specify the type of transfer entropy as well. Default is
\code{entropy = 'Shannon'}.}

\item{shuffles}{the number of shuffles used to calculate the effective
transfer entropy. Default is \code{shuffles = 100}.}

\item{type}{specifies the type of discretization applied to the observed time
series:'quantiles', 'bins' or 'limits'. Default is
\code{type = 'quantiles'}.}

\item{quantiles}{specifies the quantiles of the empirical distribution of the
respective time series used for discretization.
Default is \code{quantiles = c(5,95)}.}

\item{bins}{specifies the number of bins with equal width used for
discretization. Default is \code{bins = NULL}.}

\item{limits}{specifies the limits on values used for discretization.
Default is \code{limits = NULL}.}

\item{burn}{the number of observations that are dropped from the beginning of
the bootstrapped Markov chain. Default is \code{burn = 50}.}

\item{seed}{a seed that seeds the PRNG (will internally just call set.seed),
default is \code{seed = NULL}.}

\item{na.rm}{if missing values should be removed (will remove the values at
the same point in the other series as well). Default is \code{TRUE}.}
}
\value{
a single numerical value for the effective transfer entropy
}
\description{
Calculates the Effective Transfer Entropy for two time series
}
\examples{
# construct two time-series
set.seed(1234567890)
n <- 1000
x <- rep(0, n + 1)
y <- rep(0, n + 1)

for (i in seq(n)) {
  x[i + 1] <- 0.2 * x[i] + rnorm(1, 0, 2)
  y[i + 1] <- x[i] + rnorm(1, 0, 2)
}

x <- x[-1]
y <- y[-1]

# calculate the X->Y transfer entropy value
calc_ete(x, y)

# calculate the Y->X transfer entropy value
calc_ete(y, x)

\donttest{
  # Compare the results
  # even with the same seed, transfer_entropy might return slightly different
  # results from calc_ete
  calc_ete(x, y, seed = 123)
  calc_ete(y, x, seed = 123)
  transfer_entropy(x, y, nboot = 0, seed = 123)
}
}
\seealso{
\code{\link{calc_te}} and\code{\link{transfer_entropy}}
}
