\name{NCEP.vis.area}
\alias{NCEP.vis.area}

\title{ Visualize Weather Data on a Map }

\description{ This function creates a filled contour map from weather data.  It visualizes data from a single layer (i.e. timestep) of a data array as returned by \code{\link{NCEP.gather}} or a single aggregated layer as returned by \code{\link{NCEP.aggregate}} } 

\usage{
NCEP.vis.area(wx.data, layer=1, show.pts=TRUE, draw.contours=TRUE,
    cols=heat.colors(64), transparency=.5, axis.args=NULL, map.args=NULL,
    grid.args=NULL, title.args=NULL, interp.loess.args=NULL, 
    image.plot.args=NULL, contour.args=NULL, points.args=NULL)
}

\arguments{
  \item{wx.data}{ A 3-D array of weather data as returned by \code{\link{NCEP.gather}} }
  \item{layer}{ Either a numerical indication of the layer (default is the first layer) or a character expression of the datetime of a particular layer. }
  \item{show.pts}{ Logical.  Should the points at which data were obtained be plotted? }
  \item{draw.contours}{ Logical.  Should the map include contour lines? }
  \item{cols}{ A vector of colors such as that generated by \code{\link[grDevices]{rainbow}}, \code{\link[grDevices]{heat.colors}}, \code{\link[grDevices]{topo.colors}}, \code{\link[grDevices]{terrain.colors}}, or similar functions indicating the colors of the filled contours on the map. }  
  \item{transparency}{ A numeric value between 0 and 1 indicating the transparency of the filled contours on the map.}
  \item{axis.args}{ A list of arguments controlling the drawing of axes. See \code{\link{axis}} for acceptable arguments and the examples below for a demonstration. }
  \item{map.args}{ A list of arguments controlling the drawing of the map. See \code{\link[maps]{map}} for acceptable arguments and the examples below for a demonstration. }
  \item{grid.args}{ A list of arguments controlling the drawing of the lat/long grid lines. See \code{\link{abline}} for acceptable arguments and the examples below for a demonstration. }
  \item{title.args}{ A list of arguments controlling the how titles and axis lables are written. See \code{\link{title}} for acceptable arguments and the examples below for a demonstration. }
  \item{interp.loess.args}{ A list of arguments controlling the interpolation between grid points. See \code{\link[tgp]{interp.loess}} for acceptable arguments and the examples below for a demonstration. }
  \item{image.plot.args}{ A list of arguments controlling the plotting of the filled contour surface, the color-bar legend, and the legend axis and labels. See \code{\link[fields]{image.plot}} for acceptable arguments and the examples below for a demonstration. }
  \item{contour.args}{ A list of arguments controlling the drawing of contour lines. See \code{\link{contour}} for acceptable arguments and the examples below for a demonstration. }
  \item{points.args}{ A list of arguments controlling the plotting of grid points. See \code{\link{points}} for acceptable arguments and the examples below for a demonstration. }
 }

\details{
If the specification of \code{layer} is not numeric, it must specify a layer by a character expression of its datetime using the format \code{"\%Y-\%m-\%d \%H"}.
Leading zeros in the month, day, and hour components must be given (e.g. \code{"2006-01-01 06"}) 

If the data array has been aggregated in some way, the datetime dimension may no longer contain information on one or more temporal component: year, month, day, or hour. 
In this case, replace missing datetime componets with \code{"XX"} or \code{"XXXX"} (e.g. \code{"XXXX-01-XX XX"}) when specifying a layer by a character expression of its datetime. 
Use \code{\link[base]{dimnames}} to see the datetime dimension labels. 

If the geographical area to be plotted contains no landmass, the plot will exhibit a small unfilled border.

Most of the components of a plot produced by this function can be controlled by supplying a list of arguments to the embedded function that produces the particular component of the plot.
For example, the text and size of the plot's title can be controlled by specifying a list of acceptable arguments to \code{title.args}.
Similarly, the axes, map, and grid lines are controlled by specifying a list of acceptable arguements to \code{axis.args}, \code{map.args}, and \code{grid.args}, respectively.
Through the argument \code{image.plot.args} the user can control the plotting of the filled contour surface, the color-bar legend, and the color-bar's title and axis labels.
A list of arguments passed to \code{interp.loess.args} controls the interpolation of the filled-contour surface.
See the examples below for a demonstration of how to apply these different arguments.
}

\value{ A plot is produced.  No data are returned. }

\references{ Kemp, M. U., van Loon, E. E., Shamoun-Baranes, J., and Bouten, W. 2011. RNCEP:global weather and climate data at your fingertips. -- Ecography submitted. }

\author{ Michael U. Kemp \email{M.U.Kemp@UvA.nl} }

\examples{
library(RNCEP)
## Retrieve data for a specified spatial and temporal extent ##
wx.extent <- NCEP.gather(variable = "air", level=850,
    months.minmax = 11, years.minmax = 2000,
    lat.southnorth = c(50, 60), lon.westeast = c(0, 10),
    reanalysis2 = FALSE, return.units = TRUE)

## Visualize the first layer (i.e. first timestep) of the
## variable on a map
## Note how to specify the plot's title
## Note also the adjustment of the Kernal span argument for 
## 		interpolation using interp.loess.args
NCEP.vis.area(wx.data=wx.extent, layer=1, show.pts=TRUE, draw.contours=TRUE,
    cols=heat.colors(64), transparency=.6, title.args=list(main="Example"),
    interp.loess.args=list(span=.75))

## Now visualize a particular layer by specifying its datetime ##
NCEP.vis.area(wx.data=wx.extent, layer='2000-11-01 18', show.pts=TRUE,
    draw.contours=TRUE, cols=terrain.colors(64), transparency=.6,
    title.args=list(main="Example: select layer by datetime"),
    interp.loess.args=list(span=0.5))
	
## Now produce the same graph as above ##
## This time, label the color-bar legend ##
NCEP.vis.area(wx.data=wx.extent, layer='2000-11-01 18', show.pts=TRUE,
    draw.contours=TRUE, cols=terrain.colors(64), transparency=.6,
    title.args=list(main="Example: select layer by datetime"),
    interp.loess.args=list(span=0.5), 
    image.plot.args=list(legend.args=list(text='Kelvin', cex=1.15, padj=-1, adj=-.25)))

## Now produce the same graph as above ##
## This time, explicitly specify the size and location 
## of the color-bar legend using the smallplot argument 
## in the list of image.plot.args ##
NCEP.vis.area(wx.data=wx.extent, layer='2000-11-01 18', show.pts=TRUE,
    draw.contours=TRUE, cols=terrain.colors(64), transparency=.6,
    title.args=list(main="Example: select layer by datetime"),
    interp.loess.args=list(span=0.5), 
    image.plot.args=list(legend.args=list(text='Kelvin', cex=1.15, padj=-1, adj=-.25),
        smallplot=c(0.8475, 0.875, 0.20, 0.8)))

###########################################################
## This function can also show a layer after aggregation ##
###########################################################
## Calculate the average hourly temperature from the data
## obtained above ##
wx.ag <- NCEP.aggregate(wx.data=wx.extent, YEARS=FALSE, MONTHS=FALSE,
    HOURS=TRUE, DAYS=FALSE, fxn='mean')

## Now plot the mean temperature at midnight ##
## Note the adjustment of axis labels
## Note also the adjustment of the point type
NCEP.vis.area(wx=wx.ag, layer=1, interp.loess.args=list(span=0.5),
    title.args=list(main='Example: aggregated layer', xlab='Long [degrees]',
    ylab='Lat [degrees]'), points.args=list(pch=19))

## Now produce the same plot as above ##
## This time, change the font size used in the
## contour labels ##
NCEP.vis.area(wx=wx.ag, layer=1, interp.loess.args=list(span=0.5),
    title.args=list(main='Example: aggregated layer', xlab='Long [degrees]',
    ylab='Lat [degrees]'), points.args=list(pch=19),
    contour.args=list(labcex=.6))
	
## Notice how you can plot an aggregated layer by specifying
## it explicitly ##
NCEP.vis.area(wx=wx.ag, layer="XXXX-XX-XX 18", interp.loess.args=list(span=0.5),
    title.args=list(main='Example: aggregated layer', xlab='Long [degrees]',
    ylab='Lat [degrees]'), points.args=list(pch=19),
    contour.args=list(labcex=.6))
}