
#' @useDynLib RMVL

MVL_SMALL_LENGTH<-1024

#' Open an MVL file
#'
#' Open an MVL format file for reading and/or writing.
#'
#' MVL stands for "Mapped vector library" and is a file format designed for efficient memory mapped access. 
#' An MVL file can be much larger than physical memory of the machine.
#'
#' \code{mvl_open} returns a handle that can be used to access MVL files. Files opened read-only are memory mapped and do not use a file descriptor, and thus are not
#' subject to limits on the number of open files.
#' Files opened for writing data do use a file descriptor.
#' Once opened for read access the data can be accessed using usual R semantics for lists, data.frames and arrays.
#'
#' @param filename  path to file.
#' @param append specify TRUE when you intend to write data into the file
#' @param create when TRUE create file if it did not exist
#' @return handle to opened MVL file
#' @examples
#' \dontrun{
#' M1<-mvl_open("test1.mvl", append=TRUE, create=TRUE)
#' mvl_write_object(M1, data.frame(x=1:2, y=rnorm(2)), "test_frame")
#' mvl_close(M1)
#'
#' M2<-mvl_open("test1.mvl")
#' print(names(M2))
#' print(M2["test_frame"])
#' mvl_close(M2)
#'
#' M3<-mvl_open("test2.mvl", append=TRUE, create=TRUE)
#' L<-list()
#' df<-data.frame(x=1:1e6, y=rnorm(1e6), s=rep(c("a", "b"), 5e5))
#' L[["x"]]<-mvl_write_object(M3, df, drop.rownames=TRUE)
#' L[["description"]]<-"Example of large data frame"
#' mvl_write_object(M3, L, "test_object")
#' mvl_close(M3)
#'
#' M4<-mvl_open("test2.mvl")
#' print(names(M4))
#' L<-M4["test_object"]
#' print(L)
#' print(L[["x"]][1:20,])
#' mvl_object_stats(L[["x"]])
#' # If you need to get the whole x, one can use L[["x"]][]
#' mvl_close(M4)
#' }
#' @export
#'
mvl_open<-function(filename, append=FALSE, create=FALSE) {
	MVLHANDLE<-list(handle=.Call("mmap_library", as.character(filename), as.integer(ifelse(append, 1, 0)+ifelse(create, 2, 0))))
	class(MVLHANDLE)<-"MVL"
	MVLHANDLE[["directory"]]<-mvl_get_directory(MVLHANDLE)
	return(MVLHANDLE)
	}
	
#' Close MVL file
#'
#' Closes MVL file releasing all resources.
#
#' For read-only files the memory is unmapped, reducing the virtual memory footprint.
#' For files opened for writing the directory is written out, so it is important to calls \code{mvl_close} or the newly written file will be corrupt.
#'
#' @param MVLHANDLE - handle to opened MVL file as generated by mvl_open()
#' @return None
#' @export
#'
mvl_close<-function(MVLHANDLE) {
	if(!inherits(MVLHANDLE, "MVL")) stop("not an MVL object")
	.Call("close_library", MVLHANDLE[["handle"]])

	return(invisible(NULL))
	}
	
mvl_get_directory<-function(MVLHANDLE) {
	if(!inherits(MVLHANDLE, "MVL")) stop("not an MVL object")
	return(.Call("get_directory", MVLHANDLE[["handle"]]))
	}

mvl_get_vectors<-function(MVLHANDLE, offsets, raw=FALSE) {
	if(!inherits(MVLHANDLE, "MVL")) stop("not an MVL object")
	if(!inherits(offsets, "MVL_OFFSET"))stop("not an MVL offset")
	if(raw)
		return(.Call("read_vectors_raw", MVLHANDLE[["handle"]], offsets))
		else
		return(.Call("read_vectors", MVLHANDLE[["handle"]], offsets))
	}
	
mvl_write_vector<-function(MVLHANDLE, x, metadata.offset=NULL) {
	if(!inherits(MVLHANDLE, "MVL")) stop("not an MVL object")
	if(!is.null(metadata.offset) && !inherits(metadata.offset, "MVL_OFFSET"))stop("not an MVL offset")
	if(class(x)=="factor")x<-as.character(x)
	type<-attr(x, "MVL_TYPE", exact=TRUE)
	if(is.null(type)) {
		type<-switch(class(x), raw=1, numeric=5, integer=2, MVL_OFFSET=100, character=10000, -1)
		if(type<0 && class(x) %in% c("array", "matrix"))type<-switch(typeof(x), double=5, integer=2, -1)
		}
	if(type>0) {
		return(.Call("write_vector", MVLHANDLE[["handle"]], as.integer(type), x, metadata.offset)) 
		}
	stop("Could not write vector with class ", class(x))
	}

mvl_write_string<-function(MVLHANDLE, x, metadata.offset=NULL) {
	if(!inherits(MVLHANDLE, "MVL")) stop("not an MVL object")
	if(!is.null(metadata.offset) && !inherits(metadata.offset, "MVL_OFFSET"))stop("not an MVL offset")
	x<-as.character(x)
	if(length(x)!=1)stop("requires a single string as argument")
	return(.Call("write_vector", MVLHANDLE[["handle"]], as.integer(10001), x, metadata.offset)) 
	}
	
mvl_write_object_metadata<-function(MVLHANDLE, x, drop.rownames=FALSE) {
	n<-mvl_write_string(MVLHANDLE, "MVL_LAYOUT")
	o<-mvl_write_string(MVLHANDLE, "R")
	if(!is.null(dim(x))) {
		n<-c(n, mvl_write_string(MVLHANDLE, "dim"))
		o<-c(o, mvl_write_vector(MVLHANDLE, dim(x)))
		}
	if(!is.null(class(x)) && !(class(x) %in% c("raw", "numeric", "integer"))) {
		n<-c(n, mvl_write_string(MVLHANDLE, "class"))
		o<-c(o, mvl_write_string(MVLHANDLE, class(x)))
		}
	if(!is.null(names(x))) {
		n<-c(n, mvl_write_string(MVLHANDLE, "names"))
		o<-c(o, mvl_write_vector(MVLHANDLE, names(x)))
		}
	if(!is.null(rownames(x)) && !drop.rownames) {
		n<-c(n, mvl_write_string(MVLHANDLE, "rownames"))
		o<-c(o, mvl_write_vector(MVLHANDLE, rownames(x)))
		}
	if(is.null(n))return(NULL)
	ofs<-c(n, o)
	class(ofs)<-"MVL_OFFSET"
	return(mvl_write_vector(MVLHANDLE, ofs))
	}

#' Write R objects into MVL file
#'
#' @param MVLHANDLE a handle to MVL file produced by mvl_open()
#' @param x a suitable R object (vector, array, list, data.frame)
#' @param name if specified add a named entry to MVL file directory
#' @param drop.rownames set to TRUE to prevent rownames from being written
#' @return and object of class MVL_OFFSET that describes an offset into this MVL file. MVL offsets are vectors and can be concatenated. They can be written to MVL file directly, or as part of another object such as list.
#'  
#' @export
#'
mvl_write_object<-function(MVLHANDLE, x, name=NULL, drop.rownames=FALSE) {
	#cat("Writing", class(x), typeof(x), "\n")
	metadata<-mvl_write_object_metadata(MVLHANDLE, x, drop.rownames=drop.rownames)
	if(class(x) %in% c("numeric", "character", "integer", "factor", "raw", "array", "matrix")) {
		offset<-mvl_write_vector(MVLHANDLE, x, metadata)
		if(!is.null(name))mvl_add_directory_entries(MVLHANDLE, name, offset)
		return(invisible(offset))
		}
	if(class(x) %in% c("list", "data.frame")) {
		v<-unlist(lapply(x, function(x){return(mvl_write_object(MVLHANDLE, x))}))
		class(v)<-"MVL_OFFSET"
		offset<-mvl_write_vector(MVLHANDLE, v, metadata)
		if(!is.null(name))mvl_add_directory_entries(MVLHANDLE, name, offset)
		return(invisible(offset))
		}
	if(class(x) == "MVL_OFFSET") {
		# Already written
		return(invisible(x))
		}
	stop("Could not write object with class ", class(x))
	}
	
mvl_flatten_string<-function(v) {
	return(unlist(lapply(v, function(x){return(x[[1]])})))
	}

mvl_read_metadata<-function(MVLHANDLE, metadata_offset) {
	metadata<-mvl_read_object(MVLHANDLE, metadata_offset, recurse=FALSE)
	if(!is.null(metadata)) {
		n<-metadata[1:(length(metadata)/2)]
		metadata<-metadata[(length(metadata)/2+1):length(metadata)]
		names(metadata)<-unlist(n)
		}
	return(metadata)
	}
	
#' Return MVL object properties
#'
#' Provide detailed information on stored MVL object without retrieving it
#'
#' This function is given either an MVL handle and an offset in MVL file to examine, or just a single parameter of class MVL_OBJECT that contains
#'  both handle and offset
#'
#' The function returns list of object parameters including total number of element, element type (as used by libMVL) and a pointer to the underlying data
#' The pointer is passed cast to double and can be used with custom C code, for example by using package inline.
#'
#' @param MVLHANDLE either a handle provided by mvl_open() or and MVL object such as produced by indexing operators
#' @param offset offset to the object which properties are to be retrieved
#' @return list with object properties
#' @export
#'
mvl_object_stats<-function(MVLHANDLE, offset=NULL) {
	if(!inherits(MVLHANDLE, "MVL") && !inherits(MVLHANDLE, "MVL_OBJECT")) stop("not an MVL object")
	if(is.null(offset) && inherits(MVLHANDLE, "MVL_OBJECT"))offset<-unclass(MVLHANDLE)[["offset"]]
	if(!inherits(offset, "MVL_OFFSET"))stop("not an MVL offset")
	
	L<-list(handle=MVLHANDLE[["handle"]], 
		offset=offset, 
		metadata_offset=.Call("read_metadata", MVLHANDLE[["handle"]], offset),
		length=.Call("read_lengths", MVLHANDLE[["handle"]], offset), 
		type=.Call("read_types", MVLHANDLE[["handle"]], offset), 
		data_pointer=.Call("get_vector_data_ptr", MVLHANDLE[["handle"]], offset)
		)
	return(L)
	}
	
mvl_read_object<-function(MVLHANDLE, offset, idx=NULL, recurse=FALSE, raw=FALSE) {
	if(!inherits(MVLHANDLE, "MVL") && !inherits(MVLHANDLE, "MVL_OBJECT")) stop("not an MVL object")
	if(!inherits(offset, "MVL_OFFSET"))stop("not an MVL offset")
	if(is.na(offset))return(NA)
	if(offset==0)return(NULL)
	metadata_offset<-.Call("read_metadata", MVLHANDLE[["handle"]], offset)
	metadata<-mvl_read_metadata(MVLHANDLE, metadata_offset)
	cl<-metadata[["class"]]
	
	if(any(metadata[["MVL_LAYOUT"]]=="R") && !recurse && !is.null(cl) && (cl=="data.frame")) {
		L<-list(handle=MVLHANDLE[["handle"]], offset=offset, length=.Call("read_lengths", MVLHANDLE[["handle"]], offset), type=.Call("read_types", MVLHANDLE[["handle"]], offset),metadata_offset=metadata_offset)
		L[["metadata"]]<-metadata
		class(L)<-"MVL_OBJECT"
		return(L)
		}
	
	if(is.null(idx)) {
		if(raw) 
			vec<-.Call("read_vectors_raw", MVLHANDLE[["handle"]], offset)[[1]]
			else
			vec<-.Call("read_vectors", MVLHANDLE[["handle"]], offset)[[1]]
		} else {
		if(raw)
			vec<-.Call("read_vectors_idx_raw_real", MVLHANDLE[["handle"]], offset, idx[[1]])[[1]]
			else
			vec<-.Call("read_vectors_idx_real", MVLHANDLE[["handle"]], offset, idx[[1]])[[1]]
		}
	if(inherits(vec, "MVL_OFFSET")) {
		lengths<-.Call("read_lengths", MVLHANDLE[["handle"]], vec)
		if(recurse) {
			vec<-lapply(vec, function(x){class(x)<-"MVL_OFFSET" ; return(mvl_read_object(MVLHANDLE, x, recurse=TRUE, raw=raw))})
		 } else {
			Fsmall<-lengths<MVL_SMALL_LENGTH
			vec[Fsmall]<-lapply(vec[Fsmall], function(x){class(x)<-"MVL_OFFSET" ; return(mvl_read_object(MVLHANDLE, x, recurse=FALSE, raw=raw))})
			vec[!Fsmall]<-lapply(vec[!Fsmall], function(x) {
				class(x)<-"MVL_OFFSET"
				L<-list(handle=MVLHANDLE[["handle"]], offset=x, length=.Call("read_lengths", MVLHANDLE[["handle"]], x), type=.Call("read_types", MVLHANDLE[["handle"]], x),metadata_offset=.Call("read_metadata", MVLHANDLE[["handle"]], x))
				L[["metadata"]]<-mvl_read_metadata(MVLHANDLE, L[["metadata_offset"]])
				class(L)<-"MVL_OBJECT"
				return(L)} 
				)
			}
		}
#	attr(vec, "metadata")<-metadata
	if(any(metadata[["MVL_LAYOUT"]]=="R")) {
		if(!is.null(cl)) {
			if(cl!="data.frame" && !is.null(metadata[["dim"]]))dim(vec)<-metadata[["dim"]]
			if(cl=="factor" || cl=="character") {
				vec<-mvl_flatten_string(vec)
				if(cl=="factor")vec<-as.factor(vec)
				}
			#if(cl=="data.frame" && any(unlist(lapply(vec, class))=="MVL_OBJECT"))cl<-"MVL_OBJECT"
			class(vec)<-cl
			}
		if(!is.null(metadata[["names"]]))names(vec)<-mvl_flatten_string(metadata[["names"]])
		rn<-metadata[["rownames"]]
		if(!is.null(rn) && class(rn)!="MVL_OBJECT")rownames(vec)<-mvl_flatten_string(metadata[["rownames"]])
			else
		if(!is.null(cl) && cl=="data.frame")rownames(vec)<-1:(metadata[["dim"]][1])
		}
	return(vec)
	}
	
#' Add entries to MVL directory
#' 
#' Add one or more entries to MVL directory
#'
#' This function is used to expand MVL directory. The offsets must be created by calling \code{mvl_write_object} on the same handle.
#' Note that \code{mvl_write_object} has an optional parameter \code{name} that will add an entry when specified.
#' Thus this function is meant for special circumstances, such as creating multiple entries in the directory that point to the same offset
#'
#' @param MVLHANDLE handle to open MVL file created by \code{mvl_open}
#' @param tag a vector of one or more character tags
#' @param offsets a vector of MVL_OFFSET objects, one per tag, created by \code{mvl_write_object}
#'
#' @export
mvl_add_directory_entries<-function(MVLHANDLE, tag, offsets) {
	if(!inherits(MVLHANDLE, "MVL")) stop("not an MVL object")
	if(!inherits(offsets, "MVL_OFFSET"))stop("not an MVL offset")
	return(.Call("add_directory_entries", MVLHANDLE[["handle"]], as.character(tag), offsets))
	}
	
#' @export
`[.MVL_OFFSET`<-function(x, y) {
	z<-unclass(x)[y]
	class(z)<-"MVL_OFFSET"
	return(z)
	}

#' @export
`[[.MVL_OFFSET`<-function(x, y) {
	z<-unclass(x)[[y]]
	class(z)<-"MVL_OFFSET"
	return(z)
	}
	
# We are exporting plain function as well, so one can list its source code from command line
#' MVL handle subscription operator
#'
#' Retrieve objects stored in mappable vector library
#
#' See \code{mvl_open} for example.
#'
#' @param MVLHANDLE - handle to opened MVL file as generated by \code{mvl_open}
#' @param y - name of object to retrieve
#' @param raw - request to return data in raw format when it does not map exactly to R data types. 
#' @return Stored object
#' @export [.MVL
#' @export
`[.MVL`<-function(MVLHANDLE, y, raw=FALSE) {
	if(!inherits(MVLHANDLE, "MVL")) stop("not an MVL object")
	
	if(is.factor(y))y<-as.character(y)
	
	if(is.character(y)) {
		offset<-MVLHANDLE[["directory"]][[y]]
		if(is.null(offset))return(NULL)
		
		obj<-list(handle=MVLHANDLE[["handle"]], offset=offset, 
			length=.Call("read_lengths", MVLHANDLE[["handle"]], offset), 
			type=.Call("read_types", MVLHANDLE[["handle"]], offset),
			metadata_offset=.Call("read_metadata", MVLHANDLE[["handle"]], offset))
		obj[["metadata"]]<-mvl_read_metadata(MVLHANDLE, obj[["metadata_offset"]])
		class(obj)<-"MVL_OBJECT"
		
		if(obj[["length"]]<MVL_SMALL_LENGTH)obj<-mvl_read_object(MVLHANDLE, obj[["offset"]], recurse=FALSE, raw=raw)
		
		return(obj)
		}
	stop("Cannot process ", y, " class=", class(y))
	}
	
#' MVL handle subscription operator
#'
#' Retrieve objects stored in the library. Unlike for R lists the match on name is always exact. 
#'
#' @param MVLHANDLE - handle to opened MVL file as generated by \code{mvl_open}
#' @param name - name of object to retrieve
#' @return Stored object
#' @export $.MVL
#' @export
`$.MVL`<-function(MVLHANDLE, name) {
	return(MVLHANDLE[name])
	}
	
#' Print MVL directory
#' 
#' @param x handle to MVL file as created by \code{mvl_open}
#' @return character vector of names present in the directory
#'
#' @export
names.MVL<-function(x) {
	if(!inherits(x, "MVL")) stop("not an MVL object")
	return(names(x[["directory"]]))
	}
	
#' Print MVL 
#' 
#' @param x handle to MVL file as created by \code{mvl_open}
#' @param \ldots not used.
#' @return invisible(MVLHANDLE)
#'
#' @export
print.MVL<-function(x, ...) {
	if(!inherits(x, "MVL")) stop("not an MVL object")
	if(length(x[["directory"]])< MVL_SMALL_LENGTH)
		cat("MVL(handle ", x[["handle"]], " directory with ", length(x[["directory"]]), " entries, [c(\"", paste0(names(x[["directory"]]), collapse="\", \""), "\")])\n", sep="")
		else
		cat("MVL(handle ", x[["handle"]], " directory with ", length(x[["directory"]]), " entries, [c(\"", paste0(names(x[["directory"]])[1:MVL_SMALL_LENGTH], collapse="\", \""), "\")])\n", sep="")
	invisible(x)
	}
	
MVL_TYPE_NAME<-list("UINT8", "INT32", "INT64", "FLOAT", "DOUBLE")
MVL_TYPE_NAME[[100]]<-"OFFSET64"
MVL_TYPE_NAME[[101]]<-"CSTRING"
	
mvl_type_name<-function(x) {
	y<-lapply(MVL_TYPE_NAME[x], function(xx){if(is.null(xx))return(NA); return(xx)})
	return(unlist(y))
	}
	
#' Print MVL object
#
#' This is a convenience function for displaying MVL_OBJECTs.
#' 
#' @param x MVL_OBJECT as retrieved by subscription operators
#' @param \ldots not used.
#' @param small_length do not list more than this number of columns in data frames
#' @return invisible(obj)
#'
#' @export
print.MVL_OBJECT<-function(x, ..., small_length=10) {
	obj<-x
	object_class<-obj[["metadata"]][["class"]]
	if(is.null(object_class) || (object_class %in% c("numeric", "integer"))) {
		cat("MVL_OBJECT(", mvl_type_name(obj[["type"]]), " length ", obj[["length"]], ")\n", sep="")
		} else
	if(object_class %in% c("data.frame", "array", "matrix")) {
		od<-obj[["metadata"]][["dim"]]
		if(is.null(od))od<-obj[["length"]]
		
		nm<-obj[["metadata"]][["names"]]
		if(length(nm)<1 || length(od)!=2) 
			cat("MVL_OBJECT(", mvl_type_name(obj[["type"]]), " ", object_class, " ", paste0(od, collapse="x"), ")\n", sep="")
			else
		if(length(nm)< small_length) 
			cat("MVL_OBJECT(", mvl_type_name(obj[["type"]]), " ", object_class, " ", paste0(od, collapse="x"), " [,c(\"", paste0(nm, collapse="\", \""), "\")] )\n", sep="")
			else
			cat("MVL_OBJECT(", mvl_type_name(obj[["type"]]), " ", object_class, " ", paste0(od, collapse="x"), " [,c(\"", paste0(nm[1:small_length], collapse="\", \""), "\", ...)] )\n", sep="")
		} else {
		cat("MVL_OBJECT(", mvl_type_name(obj[["type"]]), " ", object_class, ")\n", sep="")
		}
	invisible(obj)
	}
	
#' Obtain dimensions of MVL object
#' 
#' @param x MVL_OBJECT as retrieved by subscription operators
#' @return object dimensions, or NULL if not present
#'
#' @export
dim.MVL_OBJECT<-function(x) {
	if(is.null(x[["metadata"]]))stop("Malformed MVL_OBJECT: missing metadata")
	return(x[["metadata"]][["dim"]])
	}
	
#' Obtain length of MVL object
#' 
#' @param x MVL_OBJECT as retrieved by subscription operators
#' @return object length as stored in MVL file. This is the total length of object for arrays, and number of columns for data frames.
#'
#' @export
length.MVL_OBJECT<-function(x) {
	return(x[["length"]])
	}
	
# We are exporting plain function as well, so one can list its source code from command line
#' MVL object subscription operator
#'
#' Retrieve objects stored in mappable vector library. Large nested objects are returned as instances of MVL_OBJECT to delay access until needed.
#
#' See \code{mvl_open} for example.
#'
#' @param obj - MVL object retrieved by subscription of MVL library or other objects
#' @param i - optional index.
#' @param \ldots optional additional indices for multidimensional arrays and data frames
#' @param drop - whether to drop dimensionality, such as done with R array or data frames
#' @param raw - request to return data in raw format when it does not map exactly to R data types. 
#' @param recurse - force recursive conversion to pure R objects. 
#' @return Stored object
#' @export [.MVL_OBJECT
#' @export
`[.MVL_OBJECT`<-function(obj, i, ..., drop=NULL, raw=FALSE, recurse=FALSE) {
	if(missing(i) && ...length()==0) {
		return(mvl_read_object(obj, unclass(obj)[["offset"]], recurse=recurse, raw=raw))
		}
	#cat("obj class ", obj[["metadata"]][["class"]], "\n")
	object_class<-obj[["metadata"]][["class"]]
	if(is.null(object_class))object_class<-"NULL"
	if(object_class=="data.frame") {
		if(...length()>1)stop("Object", obj, "has only two dimensions")
		n<-obj[["metadata"]][["names"]]
		if(missing(i)) {
			i<-1:(obj[["metadata"]][["dim"]][1])
			}
		if(is.logical(i)) {
			i<-which(i)
			}
		if(...length()<1 || missing(..1)) {
			j<-1:length(n)
			} else {
			j<-..1
			if(is.logical(j)) {
				j<-which(j)
				} else
			if(is.character(j) || is.factor(j)) {
				if(is.factor(j))j<-as.character(j)
				j0<-match(j, n)
				if(any(is.na(j0)))
					stop("Unknown columns ", j[is.na(j0)])
				j<-j0
				}
			n<-n[j]
			}
		if(raw)
			ofs<-.Call("read_vectors_raw", obj[["handle"]], obj[["offset"]])[[1]][j]
			else
			ofs<-.Call("read_vectors", obj[["handle"]], obj[["offset"]])[[1]][j]
#		vec<-.Call("read_vectors", obj[["handle"]], ofs)
		df<-lapply(ofs, function(x){class(x)<-"MVL_OFFSET" ; return(mvl_read_object(obj, x, idx=list(as.numeric(i-1))))})
		names(df)<-n
		class(df)<-"data.frame"
		if(dim(df)[2]==1 && !is.null(drop) && drop)return(df[,1])
		
		rn<-obj[["metadata"]][["rownames"]]
		if(class(rn)=="MVL_OBJECT") {
			rn<-mvl_flatten_string(rn[i, recurse=TRUE])
			rownames(df)<-rn
			} else {
			rownames(df)<-1:(length(i))
			}
		return(df)
		}
	if(object_class %in% c("array", "matrix")) {
		od<-obj[["metadata"]][["dim"]]
		if(is.null(od))od<-obj[["length"]]
		
		if(missing(i)) {
			d<-od[1]
			idx<-0:(od[1]-1)
			} else {
			if(is.logical(i))i<-which(i)
			d<-length(i)
			idx<-i-1
			}
		mult<-1
		
		if(...length()+1!=length(od))stop("Array dimension is ", length(od), " but ", ...length()+1, " indices given")
		
		if(...length()>0) {
			for(j in 1:...length()) {
				ii<-NULL
				try({ii<-...elt(j)}, silent=TRUE)
				if(is.null(ii)) {
					d<-c(d, od[j+1])
					ii<-1:od[j+1]
					} else {
					if(is.logical(ii))ii<-which(ii)
					d<-c(d, length(ii))
					}
				mult<-mult*od[j]
				idx<-outer(idx, (ii-1)*mult, FUN="+")
				}
			}
		if(raw)
			vec<-.Call("read_vectors_idx_raw_real", obj[["handle"]], obj[["offset"]], as.numeric(idx))[[1]]
			else
			vec<-.Call("read_vectors_idx_real", obj[["handle"]], obj[["offset"]], as.numeric(idx))[[1]]
		
		if(is.null(drop) || drop==TRUE) {
			d<-d[d!=1]
			if(length(d)>0)dim(vec)<-d
			} else
			dim(vec)<-d
		return(vec)
		}
	if(...length()==0) {
		if(is.logical(i)) {
			i<-which(i)
			}
		if(is.factor(i))i<-as.character(i)
		if(is.character(i)) {
			if(is.null(obj$metadata$names))stop("Object has no names")
			i<-which.max(obj$metadata$names==i)
			}
		if(is.numeric(i)) {
			#print(i)
			#print(L)
#			vec<-mvl_read_object(obj, obj[["offset"]], idx=list(as.integer(i)), recurse=FALSE)
			if(raw)
				vec<-.Call("read_vectors_idx_raw", obj[["handle"]], obj[["offset"]], as.integer(i-1))[[1]]
				else
				vec<-.Call("read_vectors_idx", obj[["handle"]], obj[["offset"]], as.integer(i-1))[[1]]
#			vec<-.Call("read_vectors", obj[["handle"]], obj[["offset"]])[[1]][i]

			if(inherits(vec, "MVL_OFFSET") && length(vec)==1) {
				vec<-mvl_read_object(obj, vec, recurse=FALSE)
				} else {
				#metadata_offset<-.Call("read_metadata", obj[["handle"]], obj[["offset"]])
				#metadata<-mvl_read_metadata(obj, metadata_offset)
				#print(metadata)
# 				if(0 && any(metadata[["MVL_LAYOUT"]]=="R")) {
# 					cl<-metadata[["class"]]
# 					if(cl!="data.frame" && !is.null(metadata[["dim"]]))dim(vec)<-metadata[["dim"]]
# 					if(cl=="factor" || cl=="character") {
# 						vec<-mvl_flatten_string(vec)
# 						if(cl=="factor")vec<-as.factor(vec)
# 						}
# 						class(vec)<-cl
# 					if(!is.null(metadata[["names"]]))names(vec)<-mvl_flatten_string(metadata[["names"]])
# 					if(!is.null(metadata[["rownames"]]))rownames(vec)<-mvl_flatten_string(metadata[["rownames"]])
# 					}				
				}
			if(inherits(vec, "MVL_OFFSET") && recurse) {
				vec<-lapply(vec, function(x) {class(x)<-"MVL_OFFSET" ; return(mvl_read_object(obj, x, recurse=recurse, raw=raw)) })
				}
			return(vec)
			}
		} else {
		}
	stop("Cannot process ", obj)
	}
	
.onUnload <- function (libpath) {
  library.dynam.unload("RMVL", libpath)
}
