#' @title Creates gray-level co-occurrence matrix of RIA image
#' @export
#'
#' @description  Creates gray-level co-occurrence matrix (GLCM) from \emph{RIA_image}.
#' GLCM assesses the spatial relation of voxels to each other. By default the \emph{$modif}
#' image will be used to calculate GLCMs. If \emph{use_slot} is given, then the data
#' present in \emph{RIA_image$use_slot} will be used for calculations.
#' Results will be saved into the \emph{glcm} slot. The name of the subslot is determined
#' by the supplied string in \emph{save_name}, or is automatically generated by RIA.
#'
#' @param RIA_data_in \emph{RIA_image}, created by \code{\link[RIA]{load_dicom}}.
#'
#' @param off_right integer, indicating the number of voxels to look to the right.
#' Negative values indicate to the left.
#'
#' @param off_down integer, indicating the number of voxels to look down.
#' Negative values indicate up.
#'
#' @param off_z integer, indicating the number of voxels to look in cross plane.
#'
#' @param symmetric logical, indicating whether to create a symmetric glcm
#' by also calculating the glcm in the opposite direction (-1*off_right; -1*off_down;
#' -1*off_z), and add it to the glcm
#'
#' @param normalize logical, indicating whether to change glcm elements to relaive frequencies.
#'
#' @param use_type string, currently only "single" data processing is supported.
#'
#' @param use_orig logical, indicating to use image present in \emph{RIA_data$orig}.
#' If FALSE, the modified image will be used stored in \emph{RIA_data$modif}.
#'
#' @param use_slot string, name of slot where data wished to be used is. Use if the desired image
#' is not in the \emph{data$orig} or \emph{data$modif} slot of the \emph{RIA_image}. For example,
#' if the desired dataset is in \emph{RIA_image$dichotomized$ep_4}, then \emph{use_slot} should be
#' \emph{dichotomized$ep_4}. The results are automatically saved. If the results are not saved to
#' the desired slot, then please use \emph{save_name} parameter.
#'
#' @param save_name string, indicating the name of subslot of \emph{$glcm} to save results to.
#' If left empty, then it will be automatically determined by RIA.
#'
#' @param verbose_in logical indicating whether to print detailed information.
#' Most prints can also be suppressed using the \code{\link{suppressMessages}} function.
#'
#' @return \emph{RIA_image} containing the GLCM.
#'
#' @examples \dontrun{
#' #Dichotomize loaded image and then calculate GLCM matrix of RIA_image$modif
#' RIA_image <- dichotomize(RIA_image, bins_in = c(4, 8), equal_prob = TRUE,
#' use_orig = TRUE, write_orig = FALSE)
#' RIA_image <- glcm(RIA_image, use_orig = FALSE, verbose_in = TRUE)
#'
#' #Use use_slot parameter to set which image to use
#' RIA_image <- glcm(RIA_image, use_orig = FALSE, use_slot = "dichotomized$ep_4",
#' off_right = 2, off_down = -1, off_z = 0)
#' }

glcm <- function(RIA_data_in, off_right = 1, off_down = 0, off_z = 0, symmetric = TRUE, normalize = TRUE, use_type = "single", use_orig = FALSE, use_slot = NULL, save_name = NULL, verbose_in = TRUE)
{
  data_in_orig <- check_data_in(RIA_data_in, use_type = use_type, use_orig = use_orig, verbose_in = verbose_in)

  if(any(class(data_in_orig) != "list")) data_in_orig <- list(data_in_orig)
  list_names <- names(data_in_orig)

  for (k in 1: length(data_in_orig))
  {
  data_in <-  data_in_orig[[k]]

  data_NA <- as.vector(data_in)
  data_NA <- data_NA[!is.na(data_NA)]
  if(length(data_NA) == 0) {stop("WARNING: SUPPLIED RIA_image DOES NOT CONTAIN ANY DATA!!!")}
  if(length(dim(data_in)) < 2 | length(dim(data_in)) > 3) stop(paste0("DATA LOADED IS ", length(dim(data_in)), " DIMENSIONAL. ONLY 2D AND 3D DATA ARE SUPPORTED!"))

  dim_x <- dim(data_in)[1]
  dim_y <- dim(data_in)[2]
  dim_z <- ifelse(!is.na(dim(data_in)[3]), dim(data_in)[3], 1)

  dist <- max(abs(off_right), abs(off_down), abs(off_z))  ##maximum offset needed to increase matrix

  base_m <- array(NA, dim = c(dim_x+2*dist, dim_y+2*dist, dim_z+2*dist))
  base_m[(1+dist):(dim_x+dist), (1+dist):(dim_y+dist), (1+dist):(dim_z+dist)] <- data_in ##create base matrix containing data_in which is enlarged by dist in all directions

  shift_m <- array(NA, dim = c(dim_x+2*dist, dim_y+2*dist, dim_z+2*dist))
  shift_m[(1+(dist+off_down)):(dim_x+(dist+off_down)), (1+(dist-off_right)):(dim_y+(dist-off_right)), (1+(dist+off_z)):(dim_z+(dist+off_z))] <- data_in

  diff_m <- base_m - shift_m ##differency matrix

  adj_m <- base_m - diff_m ##matrix containing adjacency elements at given offset

  data_v <- as.vector(data_in)
  data_v <- data_v[!is.na(data_v)]
  gray_levels <- length(unique(data_v))

  glcm <- matrix(NA, nrow = gray_levels, ncol = gray_levels)

  for (i in 1:gray_levels)
  {
    for (j in 1: gray_levels)
    {
      glcm[i, j] <- sum(adj_m==i & base_m==j, na.rm = TRUE)
    }
  }
  if(symmetric) glcm <- (glcm + t(glcm))
  if(normalize) {
    if(sum(glcm) == 0) {

    } else {glcm <- glcm/sum(glcm)} }



  if(use_type == "single") {

    if(any(class(RIA_data_in) == "RIA_image") )
    {
      if(is.null(save_name)) {
        txt <- automatic_name(RIA_data_in, use_orig, use_slot)
        txt <- paste0(txt, "_", off_right, off_down, off_z)

        RIA_data_in$glcm[[txt]] <- glcm

      }
      if(!is.null(save_name)) {RIA_data_in$glcm[[save_name]] <- glcm
      }
    }
  }



  if(is.null(save_name)) {txt_name <- txt
  } else {txt_name <- save_name}
  if(verbose_in) {message(" "); message(paste0("GLCM WAS SUCCESSFULLY ADDED TO '", txt_name, "' SLOT OF RIA_image$glcm")); message(" ") }

  }

  if(any(class(RIA_data_in) == "RIA_image") ) return(RIA_data_in)
  else return(glcm)

}
