#' Gets fluorescence data vectors from \code{RDML} object
#' 
#' Gets fluorescence data vectors from \code{RDML} object for specified method
#' of experiment.
#' 
#' @param request Output from AsTable method(\link{RDML.AsTable})
#' @param dp.type Type of fluorescence data (i.e. 'adp' for qPCR or 'mdp' for
#'   melting)
#' @param long.table Output table is ready for ggplot (See \link{RDML.AsTable}
#'   for example)
#' @return \code{matrix} which contains selected fluorescence data and 
#'   additional information fromm request if \code{long.table = TRUE}.
#' @author Konstantin A. Blagodatskikh <k.blag@@yandex.ru>, Stefan Roediger 
#'   <stefan.roediger@@b-tu.de>, Michal Burdukiewicz 
#'   <michalburdukiewicz@@gmail.com> 
#' @keywords manip
#' @docType methods
#' @name RDML.GetFData
#' @rdname getfdata-method
#' @include RDML.R
#' @examples
#' \dontrun{
#' ## internal dataset BioRad_qPCR_melt.rdml (in 'data' directory)
#' ## generated by Bio-Rad CFX96. Contains qPCR and melting data.
#' ## Import without splitting by targets/types and with
#' ## custom name pattern.
#' PATH <- path.package("RDML")
#' filename <- paste(PATH, "/extdata/", "BioRad_qPCR_melt.rdml", sep ="")
#' cfx96 <- RDML$new(filename)
#' ## Select melting fluorescence data with sample.type 'unkn'.
#' library(dplyr)
#' tab <- cfx96$AsTable()
#' fdata <- cfx96$GetFData(filter(tab, sample.type == "unkn"),
#'                         dp.type = "adp")
#' ## Show names for obtained fdata
#' colnames(fdata)
#' }
RDML$set("public", "GetFData",
         function(request,
                  limits = NULL,
                  dp.type = "adp",
                  long.table = FALSE) {
           if (missing(request))
             request <- self$AsTable()
           else
             request <- data.table(request)
           if (length(unique(request$fdata.name)) != length(request$fdata.name)) {
             warning("fdata.name column has duplicates! Generating new by exp.id, run.id, react.id and target")
             request[, fdata.name := paste(exp.id, run.id, react.id, target, sep = "_"),
                     by = .(exp.id, run.id, react.id, target)]
           }
           out <- 
             request[, self$experiment[[exp.id]]$run[[run.id]]$react[[as.character(
               react.id)]]$data[[target]]$GetFData(dp.type = dp.type), by = .(fdata.name)]
           ifelse(long.table == FALSE,
             return(dcast(out,
                          as.formula(sprintf("%s ~ fdata.name",
                                             ifelse(dp.type == "adp",
                                                    "cyc", "tmp"))),
                          value.var = "fluor")),
             return(merge(request, out, by = "fdata.name"))
           )
         }, 
         overwrite = TRUE)