% Generated by roxygen2 (4.1.0): do not edit by hand
% Please edit documentation in R/LEDoptim.R
\name{LEDoptim}
\alias{LEDoptim}
\title{Optimization function for D-family, E-family and Laplace approximated ED designs}
\usage{
LEDoptim(poped.db, model_switch = NULL, ni = NULL, xt = NULL, x = NULL,
  a = NULL, bpopdescr = NULL, ddescr = NULL, maxxt = NULL,
  minxt = NULL, maxa = NULL, mina = NULL, ofv_init = 0, fim_init = 0,
  trflag = TRUE, header_flag = TRUE, footer_flag = TRUE,
  opt_xt = poped.db$settings$optsw[2], opt_a = poped.db$settings$optsw[4],
  opt_x = poped.db$settings$optsw[3], out_file = NULL, d_switch = FALSE,
  use_laplace = T, laplace.fim = FALSE,
  use_RS = poped.db$settings$bUseRandomSearch, ...)
}
\arguments{
\item{poped.db}{A PopED database.}

\item{model_switch}{Matrix defining which response a certain sampling time belongs to.}

\item{ni}{Vector defining the number of samples for each group.}

\item{xt}{\itemize{
\item \bold{******START OF INITIAL DESIGN OPTIONS**********}}
 Matrix defining the initial sampling schedule.
 Each row is a group/individual.
 If only one vector is supplied, e.g. \code{c(1,2,3,4)}, then all groups will
have the same inital design.}

\item{x}{A matrix defining the initial discrete values for the model
Each row is a group/individual.}

\item{a}{Matrix defining the initial continuous covariate values.
n_rows=number of groups, n_cols=number of covariates.
If the number of rows is one and the number of groups > 1 then all groups are assigned the
same values.}

\item{bpopdescr}{Matrix defining the fixed effects, per row (row number = parameter_number) we should have:
\itemize{
\item column 1 the type of the distribution for E-family designs (0 = Fixed, 1 = Normal, 2 = Uniform,
 3 = User Defined Distribution, 4 = lognormal and 5 = truncated normal)
\item column 2  defines the mean.
\item column 3 defines the variance of the distribution (or length of uniform distribution).
}}

\item{ddescr}{Matrix defining the diagnonals of the IIV (same logic as for the \code{bpopdescr}).}

\item{maxxt}{Matrix or single value defining the maximum value for each xt sample.  If a single value is
supplied then all xt values are given the same maximum value.}

\item{minxt}{Matrix or single value defining the minimum value for each xt sample.  If a single value is
supplied then all xt values are given the same minimum value}

\item{maxa}{Vector defining the max value for each covariate. IF a single value is supplied then
all a values are given the same max value}

\item{mina}{Vector defining the min value for each covariate. IF a single value is supplied then
all a values are given the same max value}

\item{ofv_init}{The inital OFV. If set to zero then it is computed.}

\item{fim_init}{The initial value of the FIM. If set to zero then it is computed.}

\item{trflag}{Should the optimization be output to the screen and to a file?}

\item{header_flag}{Should the header text be printed out?}

\item{footer_flag}{Should the footer text be printed out?}

\item{opt_xt}{Should the sample times be optimized?}

\item{opt_a}{Should the continuous design variables be optimized?}

\item{opt_x}{Should the discrete design variables be optimized?}

\item{out_file}{Which file should the output be directed to?  A string, a file handle using
\code{\link{file}} or \code{""} will output to the screen.}

\item{d_switch}{\itemize{
\item \bold{******START OF CRITERION SPECIFICATION OPTIONS**********}}
D-family design (1) or ED-familty design (0) (with or without parameter uncertainty)}

\item{use_laplace}{Should the Laplace method be used in calculating the expectation of the OFV?}

\item{laplace.fim}{Should an E(FIM) be calculated when computing the Laplace approximated E(OFV).  Typically
the FIM does not need to be computed and, if desired,  this calculation
is done usng the standard MC integration technique, so can be slow.}

\item{use_RS}{should the fucntion use a random search algorithm?}

\item{...}{arguments passed to \code{\link{evaluate.fim}} and \code{\link{ofv_fim}}.}
}
\description{
Optimize the objective fucntion for D-family, E-family and Laplace approximated ED designs.
Right now there is only one optimization algorithm used in this
function
\enumerate{
\item Adaptive random search. See \code{\link{RS_opt_gen}}.
}
This function takes information from the PopED database supplied as an argument.
The PopED database supplies information about the the model, parameters, design and methods to use.
Some of the arguments coming from the PopED database can be overwritten;
if they are supplied then they are used instead of the arguments from the PopED database.
}
\examples{
## Warfarin example from software comparison in:
## Nyberg et al., "Methods and software tools for design evaluation 
##   for population pharmacokinetics-pharmacodynamics studies", 
##   Br. J. Clin. Pharm., 2014. 

## Optimization using an additive + proportional reidual error to 
##   avoid sample times at very low concentrations (time 0 or very late samoples).
library(PopED)

## find the parameters that are needed to define from the structural model
ff.PK.1.comp.oral.sd.CL

## -- parameter definition function 
## -- names match parameters in function ff
sfg <- function(x,a,bpop,b,bocc){
  parameters=c(CL=bpop[1]*exp(b[1]),
               V=bpop[2]*exp(b[2]),
               KA=bpop[3]*exp(b[3]),
               Favail=bpop[4],
               DOSE=a[1])
  return(parameters) 
}

# Adding 10\% log-normal Uncertainty to fixed effects (not Favail)
bpop_vals <- c(CL=0.15, V=8, KA=1.0, Favail=1)
bpop_vals_ed_ln <- cbind(ones(length(bpop_vals),1)*4, # log-normal distribution
                         bpop_vals,
                         ones(length(bpop_vals),1)*(bpop_vals*0.1)^2) # 10\% of bpop value
bpop_vals_ed_ln["Favail",]  <- c(0,1,0)
bpop_vals_ed_ln

## -- Define initial design  and design space
poped.db <- create.poped.database(ff_file="ff.PK.1.comp.oral.sd.CL",
                                  fg_file="sfg",
                                  fError_file="feps.add.prop",
                                  bpop=bpop_vals_ed_ln, 
                                  notfixed_bpop=c(1,1,1,0),
                                  d=c(CL=0.07, V=0.02, KA=0.6), 
                                  sigma=c(0.01,0.25),
                                  groupsize=32,
                                  xt=c( 0.5,1,2,6,24,36,72,120),
                                  minxt=0,
                                  maxxt=120,
                                  a=70,
                                  mina=0,
                                  maxa=100)
# warfain ed model

\dontrun{
  
  LEDoptim(poped.db) 
  
  LEDoptim(poped.db,opt_xt=T,rsit=10) 
  
  LEDoptim(poped.db,opt_xt=T,rsit=10,d_switch=TRUE) 

  LEDoptim(poped.db,opt_xt=T,rsit=10,laplace.fim=TRUE) 
  
  LEDoptim(poped.db,opt_xt=T,rsit=10,use_laplace=FALSE) 
  
  ## testing header and footer
  LEDoptim(poped.db,opt_xt=T,rsit=10,d_switch=TRUE,
           out_file="foobar.txt") 
  
  ff <- LEDoptim(poped.db,opt_xt=T,rsit=10,d_switch=TRUE,
                 trflag=FALSE) 
  
  LEDoptim(poped.db,opt_xt=T,rsit=10,d_switch=TRUE,
           header_flag=FALSE) 
  
  LEDoptim(poped.db,opt_xt=T,rsit=10,d_switch=TRUE,
           out_file="") 
  
  LEDoptim(poped.db,opt_xt=T,rsit=10,d_switch=TRUE,
           footer_flag=FALSE) 
  
  LEDoptim(poped.db,opt_xt=T,rsit=10,d_switch=TRUE,
           footer_flag=FALSE, header_flag=FALSE) 
  
  LEDoptim(poped.db,opt_xt=T,rsit=10,d_switch=TRUE,
           footer_flag=FALSE, header_flag=FALSE,out_file="foobar.txt") 
  
  LEDoptim(poped.db,opt_xt=T,rsit=10,d_switch=TRUE,
           footer_flag=FALSE, header_flag=FALSE,out_file="") 

}
}
\seealso{
Other Optimize: \code{\link{Doptim}};
  \code{\link{RS_opt_gen}}; \code{\link{RS_opt}};
  \code{\link{a_line_search}}; \code{\link{bfgsb_min}};
  \code{\link{calc_autofocus}};
  \code{\link{calc_ofv_and_grad}}; \code{\link{mfea}};
  \code{\link{poped_optimize}}
}

