#' @rdname manage_na
#' @title Ignore or Remove NA intensities
#'
#' @description
#' Sometimes you want to keep or remove NA values in intensities to allow for spectra with 
#' varying shapes to be analyzed together or maintained in a single Open Specy object. 
#' 
#' @param x a numeric vector or an \R OpenSpecy object
#' @param lead_tail_only logical whether to only look at leading adn tailing values. 
#' @param ig_zero logical, whether to ignore both zeros and NAs
#' @param fun the name of the function you want run, this is only used if the "ignore" type is chosen.
#' @param type character of either "ignore" or "remove".
#' @param \ldots further arguments passed to \code{fun}.
#'
#' @return
#' \code{manage_na()} return logical vectors of NA locations (if vector provided) or an
#' \code{OpenSpecy} object with ignored or removed NA values.
#'
#' @examples
#' manage_na(c(NA, -1, NA, 1, 10))
#' manage_na(c(NA, -1, NA, 1, 10), lead_tail_only = FALSE)
#' manage_na(c(NA, 0, NA, 1, 10), lead_tail_only = FALSE, ig_zero = TRUE)
#' data(raman_hdpe)
#' raman_hdpe$spectra[[1]][1:10] <- NA
#' manage_na(raman_hdpe, fun = make_rel) #would normally return all NA without na.rm = TRUE but doesn't here. 
#' manage_na(raman_hdpe, type = "remove") #will remove the first 10 values we set to NA
#' 
#' @author
#' Win Cowger, Zacharias Steinmetz
#'
#' @seealso
#' \code{OpenSpecy} object to be matched with a reference library
#' \code{fill_spec()} can be used to fill NA values in Open Specy objects.
#' \code{restrict_range()} can be used to restrict spectral ranges in other ways than removing NAs.
#'
#' @export
manage_na <- function(x, ...) {
    UseMethod("manage_na")
}

#' @rdname manage_na
#' @export
manage_na.default <- function(x, lead_tail_only = TRUE, ig_zero = FALSE, ...) {

    if(all(is.na(x))) stop("All intensity values are NA, cannot remove or ignore with manage na.")
    
    if(lead_tail_only){
        na_positions <- logical(length(x))
        if(is.na(x[1])){
            criteria = TRUE
            y = 1
            while(criteria){
                if(is.na(x[y])|(ig_zero & x[y] == 0)) na_positions[y] <- TRUE 
                y = y + 1
                criteria = is.na(x[y])|(ig_zero & x[y] == 0) 
            }
        }
        if(is.na(x[length(x)])){
            criteria = TRUE
            y = length(x)
            while(criteria){
                if(is.na(x[y])|(ig_zero & x[y] == 0)) na_positions[y] <- TRUE 
                y = y - 1
                criteria = is.na(x[y])|(ig_zero & x[y] == 0) 
            }
        }
    }
    else{
        na_positions <- is.na(x)|(ig_zero & x == 0)
    }
    
    return(na_positions)
}

#' @rdname manage_na
#' @export
manage_na.OpenSpecy <- function(x, lead_tail_only = TRUE, ig_zero = FALSE, fun, type = "ignore", ...) {
    
    consistent <- x$spectra[, lapply(.SD, manage_na, 
                               lead_tail_only = lead_tail_only, 
                               ig_zero = ig_zero)] |>
        rowSums() == 0
    
    if(type == "ignore"){
        reduced <- as_OpenSpecy(x$wavenumber[consistent], x$spectra[consistent,], x$metadata) |>
            fun(...)
        
        x$spectra <- x$spectra[, lapply(.SD, as.numeric)]
        
        x$spectra[consistent,] <- reduced$spectra        
    }
    
    if(type == "remove"){
        x <- as_OpenSpecy(x$wavenumber[consistent], x$spectra[consistent,], x$metadata)
    }
    
    return(x)

}

