% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/option_valuation.R
\name{European_option_value}
\alias{European_option_value}
\title{N-factor model European options on futures contracts valuation}
\usage{
European_option_value(
  x_0,
  parameters,
  futures_maturity,
  option_maturity,
  K,
  r,
  call = FALSE,
  verbose = FALSE
)
}
\arguments{
\item{x_0}{\code{vector}. Initial values of the state variables, where the length must correspond to the number of factors specified in the parameters.}

\item{parameters}{\code{vector}. A named vector of parameter values of a specified N-factor model. Function \code{NFCP_parameters} is recommended.}

\item{futures_maturity}{\code{numeric}. Time, in years, when the underlying futures contract matures.}

\item{option_maturity}{\code{numeric}. Time, in years,  when the American option expires.}

\item{K}{\code{numeric}. Strike price of the American Option.}

\item{r}{\code{numeric}. Annualized risk-free interest rate.}

\item{call}{\code{logical}. Is the American option a call or put option?}

\item{verbose}{\code{logical}. Should additional option value information be output? see \bold{details}.}
}
\value{
The \code{European_option_value} function returns a numeric value corresponding to the present value of an option when \code{verbose = F}.
When \code{verbose = T}, \code{European_option_value} returns a list with three objects:

\tabular{ll}{

\code{option value} \tab Present value of the option. \cr

\code{annualized volatility} \tab Annualized volatility of the option. \cr

\code{parameter sensitivity} \tab Sensitivity of option value to model parameters. \cr

\code{greeks} \tab Sensitivity of option value to option parameters. \cr

}
}
\description{
Value European Option Put and Calls under the parameters of an N-factor model.
}
\details{
\loadmathjax

The \code{European_option_value} function calculates analytic expressions of the value of European call and put options on futures contracts within the N-factor model. A European option on a commodity futures contract gives the holder
the right, but not the obligation, to buy (call) or sell (put) the underlying asset at option maturity. If the European option is exercised, the option devolves into buying or selling of the underlying futures asset.

State variables (i.e., the states of the factors of an N-factor model) are generally unobservable. Filtering the commodity pricing model using term structure data will provide the most recent optimal estimates of state variables, which can then be used to forecast and value European options.

Under the assumption that future futures prices
are log-normally distributed under the risk-neutral process, there exist analytic expressions of the value of European call and put options on futures contracts. The value of a European option on a futures contract
is given by calculating the current expected futures price and the average instantaneous variance of the futures return innovations over the life of the option.

Consider a European option with strike price \mjeqn{K}{K} and a risk-free interest rate of \mjeqn{r_f}{r_f}. The option maturity is at time \mjeqn{T_0}{T_0} and futures maturity at time \mjeqn{T_1}{T_1}.
The particular model features a state vector of length \mjeqn{N}{N} (i.e., N-factors) \mjeqn{x(t)}{x(t)}

The value of a European call option would thus be:

\mjdeqn{e^{-r T_0}  E^*[max(F(x(T_0),T_0,T_1) - K, 0)]}{e^(-r T_0)  E^*[max(F(x(T_0),T_0,T_1) - K, 0)]}

The analytic solution to call and put options are given by:

Call options:
\mjdeqn{e^{-r T_0}(F(x(0), 0, T_1)  N(d_1) - KN(d_2))}{e^(-r T_0) (F(x(0), 0, T_1)  N(d_1) - KN(d_2))}

Put options:
\mjdeqn{e^{-r T_0}(KN(-d_2) - F(x(0), 0, T_1)  N(-d_1))}{e^(-r T_0) (KN(-d_2) - F(x(0), 0, T_1)  N(-d_1))}

Where:

Where:
\mjdeqn{d_1 = \frac{\ln(F/K) + \frac{1}{2} v^2}{v}}{d_1 = (ln(F/K) + 1/2 v^2) / v}

\mjdeqn{d_2 = d_1 - v}{d_2 = d_1 - v}

Parameter \mjeqn{ N(d) }{N(d)} indicates cumulative probabilities for the standard normal distribution (i.e. \mjeqn{P(Z < d)}{P(Z<d)}).

Finally, parameter \mjeqn{v}{v}, the annualized option volatility, is given by:

\mjdeqn{Var^*[\ln(F(x(T_0), T_0, T_1))] \equiv v^2 = \sum_{i.j=1} e^{(-\kappa_i + \kappa_j)(T_1 - T_0)}Cov^*(x_i(T_0), x_j(T_0))}{Var^*[ln(F(x(T_0), T_0, T_0))] = v^2 = sum_(i.j=1) e^((-kappa[i] + kappa[j])(T_1 - T_0)) Cov(x_i(T_0), x_j(T_0))}

The annualized option volatility approaches \mjeqn{\sigma_1^2 T_0}{sigma[1]^2 T_0} as both \mjeqn{T_0}{T_0} and \mjeqn{T_1}{T_1} increase, as most uncertainty about spot prices at futures contract maturity and option expiration
are a result of uncertainty about spot prices, rather than the cost of carry (Schwartz and Smith, 2000).

The presented option valuation formulas are analogous to the Black-Scholes formulas for valuing European options on stocks that do not pay dividends

When \code{verbose = T}, the \code{European_option_value} function numerically calculates the sensitivity of option prices to underlying option and model parameters. Gradients are calculated numerically through the
\code{grad} function of the \code{numDeriv} package.
}
\examples{
##Example 1 - A European 'put' option on a futures contract following 'GBM'

European_option_value(x_0 = log(20), parameters = c(mu_rn = 0.06, sigma_1 = 0.2),
                     futures_maturity = 1, option_maturity = 1,
                     K = 20, r = 0.06, call = FALSE, verbose = TRUE)

##Example 2 - A European put option under a two-factor crude oil model:

##Step 1 - Obtain current (i.e. most recent) state vector by filtering the
##two-factor oil model:
Schwartz_Smith_oil <- NFCP_Kalman_filter(parameter_values = SS_oil$two_factor,
                                        parameter_names = names(SS_oil$two_factor),
                                        log_futures = log(SS_oil$stitched_futures),
                                        dt = SS_oil$dt,
                                        futures_TTM = SS_oil$stitched_TTM,
                                        verbose = TRUE)

##Step 2 - Calculate 'call' option price:
European_option_value(x_0 = Schwartz_Smith_oil$x_t,
                     parameters = SS_oil$two_factor,
                     futures_maturity = 2,
                     option_maturity = 1,
                     K = 20,
                     r = 0.05,
                     call = FALSE,
                     verbose = FALSE)
}
\references{
Schwartz, E. S., and J. E. Smith, (2000). Short-Term Variations and Long-Term Dynamics in Commodity Prices. \emph{Manage. Sci.}, 46, 893-911.

Cortazar, G., and L. Naranjo, (2006). An N-factor Gaussian model of oil futures prices. \emph{Journal of Futures Markets: Futures, Options, and Other Derivative Products}, 26(3), 243-268.

Paul Gilbert and Ravi Varadhan (2016). numDeriv: Accurate Numerical Derivatives. R package version 2016.8-1. https://CRAN.R-project.org/package=numDeriv
}
