#' Quick plot of updated weight vs. inconsistency (fixed or elastic)
#'
#' Plot the posterior updated weight \eqn{w'(x)} under either a fixed prior mixing
#' weight (\code{0 < w < 1}) or an elastic, data-adaptive weight (\code{w > 1}, via
#' an elastic link on the Bucher Z statistic), across an \eqn{x}-range centered at \code{y_center}.
#'
#' @details
#' This function supports both Fixed-effects (FE) and Random-effects (RE) variants:
#' \itemize{
#'   \item \emph{FE (single external trial):} provide one sampling SD for \code{s_C2C1}.
#'   \item \emph{RE (multiple external trials):} provide a vector of sampling SDs for \code{s_C2C1}
#'         and the estimated between-trial variance \code{sigma_hat2}.
#' }
#'
#' @param w Scalar. If \code{w > 1}, use the elastic (data-dependent) prior with tuning
#'   parameters \code{a} and \code{b}. If \code{0 < w < 1}, use a fixed prior weight equal to \code{w}.
#' @param a,b Elastic-link parameters (used only when \code{w > 1}). Typically
#'   \code{a = log(1/t1 - 1)} and \code{b} is obtained from the closed form using \code{delta} and \code{t0}.
#' @param delta Strong-inconsistency anchor on the \eqn{x} scale; must satisfy \code{delta > 0}.
#'   Used for guides and checks in elastic mode.
#' @param t1,t0 Target levels for the elastic anchors (elastic mode only):
#'   \code{t1} is near 1 at \code{x = 0} for \code{w(Z)}, and \code{t0} is near 0 at \code{x = delta}
#'   for the posterior weight \eqn{w'(x)}. Ignored in fixed mode.
#' @param s_EC2 Positive scalar: SD of \eqn{y_{E,C2}}.
#' @param s_EC1 Positive scalar: SD of \eqn{y_{E,C1}}.
#' @param s_C2C1 Numeric: SD(s) for the \eqn{C_2:C_1} edge. Length \code{1} implies FE; length \code{> 1} implies RE.
#' @param s0_sd Positive scalar: \eqn{\sqrt{\tau_0}} for the vague NP component.
#' @param sigma_hat2 Optional nonnegative \eqn{\hat{\sigma}^2} (RE only; ignored for FE).
#' @param y_center Scalar; the direct mean is \eqn{y_{dir}(x) = y_{center} + x}. Default \code{0}.
#' @param xlim Optional numeric length-2 range for \code{x}. By default, the range covers
#'   \code{delta} in elastic mode or \code{[-1, 1]} in fixed mode.
#' @param npoints Number of grid points used to draw the curve (default \code{2001}).
#' @param lwd,lty,main,... Base-graphics options forwarded to \link[graphics:plot]{plot}.
#'
#' @return A list (invisibly):
#' \describe{
#'   \item{curve}{A \code{data.frame} with columns \code{x} and \code{w_updated}.}
#'   \item{checks}{A small \code{data.frame} with anchor diagnostics (elastic mode only).}
#' }
#'
#' @export
plot_pw_curve <- function(w, a = NA_real_, b = NA_real_,
                          delta = NA_real_, t1 = 0.99, t0 = 0.05,
                          s_EC2, s_EC1, s_C2C1, s0_sd,
                          y_center = 0, xlim = NULL, npoints = 2001,
                          lwd = 2, lty = 1, main = NULL, ...) {
  
  # ---- validate basics
  req_pos <- function(x,nm) if(!is.finite(x) || x <= 0) stop(sprintf("`%s` must be > 0.", nm), call. = FALSE)
  if (length(w) != 1L || !is.finite(w)) stop("`w` must be a single finite scalar.")
  req_pos(s_EC2,"s_EC2"); req_pos(s_EC1,"s_EC1"); req_pos(s0_sd,"s0_sd"); req_pos(delta,"delta")
  
  if (!is.finite(y_center)) y_center <- 0

  if (length(s_C2C1) == 1L) {
    mode <- "FE"
    v_1 <- 1/(1/s_EC1^2+1/s0_sd^2)
    v_2 <- 1/(1/s_C2C1[1]^2+1/s0_sd^2)
    V_ind <- v_1 + v_2
  } else {
    mode <- "RE"
    if (is.null(sigma_hat2) )
      stop("RE mode requires `sigma_hat2` (>=0)", call. = FALSE)
    if (!is.finite(sigma_hat2) || sigma_hat2 < 0) stop("`sigma_hat2` must be finite and >= 0.", call. = FALSE)
    s2_vec <- s_C2C1^2
    v2_hat <- 1 / ( sum(1 / (s2_vec + sigma_hat2))  )
    v_1 <- 1/(1/s_EC1^2+1/s0_sd^2)
    v_2 <- 1/(1/v2_hat+1/s0_sd^2)
    V_ind <- v_1 + v_2
  }
  
  Vind  <- s_EC2^2 + V_ind
  V0    <- s_EC2^2 + s0_sd^2
  s_link <- sqrt(Vind)
  

  elastic <- (w > 1)
  
  if (elastic) {
    if (!is.finite(a) || !is.finite(b)) stop("Elastic mode: `a` and `b` must be provided.")
    if (!is.finite(delta) || delta <= 0) stop("Elastic mode: `delta` must be > 0.")
  } else {
    if (!(w > 0 && w < 1)) stop("Fixed mode: `w` must satisfy 0 < w < 1.")
  }
  
  # ---- x grid
  if (is.null(xlim)) {
    xr <- if (elastic) max(1, 1.5 * delta) else 1
    xlim <- c(-xr, xr)
  } else {
    if (length(xlim) != 2L || any(!is.finite(xlim))) stop("`xlim` must be length-2 finite.")
    xlim <- sort(xlim)
  }
  npoints <- max(101L, as.integer(npoints))
  x <- seq(xlim[1], xlim[2], length.out = npoints)
  
  
  
  logit <- function(p) stats::qlogis(p)
  Zfun  <- function(x) abs(x) / s_link
  y_dir <- function(x) y_center + x
  
  K <- function(x) {
    0.5*log( V0 / Vind ) -
      0.5*( x^2 / Vind - y_dir(x)^2 / V0 )
  }
  # achieved weights
  w_prior <- function(Z) 1 / (1 + exp(a + b * log(Z + 1)))
  w_post  <- function(x) stats::plogis( stats::qlogis( w_prior(Zfun(x)) ) + K(x) )
  
  # ---- posterior weights 
  if (elastic){
    w_updated <- w_post(x)
  } else{
    w_updated <-stats::plogis( stats::qlogis(w) + K(x))
  }
  
  
  # ---- prepare checks (elastic only)
  checks <- NULL
  if (elastic) {
    # prior and posterior at anchors
    checks <- data.frame(
      Anchor   = c("Consistency (x=0)", "Consistency (x=0)",
                   "Strong inconsistency (x=δ)", "Strong inconsistency (x=δ)"),
      Quantity = c("w(Z=0)   [prior]", "w'(0)     [posterior]",
                   "w(Z=δ)   [prior]", "w'(δ)     [posterior]"),
      Target   = c(t1, NA, NA, t0),
      Achieved = c(w_prior(0), w_post(0),
                   w_prior(Zfun(delta)), w_post(delta)),
      stringsAsFactors = FALSE
    )
    checks$AbsError <- ifelse(is.na(checks$Target), NA_real_, abs(checks$Achieved - checks$Target))
  }
  
  # ---- plot
  if (is.null(main)) {
    main <- if (elastic) sprintf("Updated weight w'(x) using dynamic weight (a=%.3f, b=%.3f)", a, b)
    else           sprintf("Updated weight w'(x) using fixed weight (w=%.3f)", w)
  }
  plot(x, w_updated, type = "l", lwd = lwd, lty = lty, ylim = c(0, 1), ...,
       xlab = expression(x == y[dir] - y[ind]),
       ylab = "Updated weight w'(x)",
       main = main)
  
  if (delta>0) {abline(v = c(-delta, delta), lty = 2)}      # ±δ guides
  if (t1>0) {abline(h = t1, lty = 4)}                    # target at Z=0 (prior)
  if (t0>0) {abline(h = t0, lty = 2)}                    # target at x=δ (posterior)
    # legend("right",
    #        legend = c("w*(x)", "x=0", "x=±δ", "t1 (w at x=0)", "t0 (w' at x=δ)"),
    #        lty = c(lty, 4, 2, 4, 2), lwd = c(lwd, 1, 1, 1, 1),
    #        bty = "n", cex = 0.9)
  
  invisible(list(curve = data.frame(x = x, w_updated = w_updated),
                 checks = checks))
}