% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MortCast.R
\docType{package}
\name{MortCast-package}
\alias{MortCast}
\alias{MortCast-package}
\title{MortCast: Estimation and Projection of Age-Specific Mortality Rates}
\description{
Age-specific mortality rates are estimated and projected using
the Kannisto, Lee-Carter and related methods as described in Sevcikova et al. (2016) <doi:10.1007/978-3-319-26603-9_15>.
}
\details{
The package implements methodology described in Sevcikova et al. (2016) that is related to 
estimating and predicting age-specific mortality rates. The main functions are:
\itemize{
   \item \code{\link{cokannisto}}: Extrapolates given mortality rates into higher ages
      using the Coherent Kannisto method. The original Kannisto method 
      (with sex-independent extrapolation) is avalable in the function 
      \code{\link{kannisto}}.
   \item \code{\link{lileecarter.estimate}}: Estimates the coherent Lee-Carter parameters
      for male and female mortality rates (Li and Lee 2005), i.e. sex-independent parameters \eqn{a_x} and \eqn{k_t}, and 
      the coherent parameter \eqn{b_x}. In addition, it 
      computes the ultimate \eqn{b^u_x} for rotation (Li et al. 2013). 
      The underlying sex-independent estimation is implemented in the function \code{\link{leecarter.estimate}}.
   \item \code{\link{mortcast}}: Using estimated coherent Lee-Carter parameters 
      and given future sex-specific life expectancies,
      it projects age-specific mortality rates, while (by default) rotating the \eqn{b_x} 
      parameter as described in Li et al. (2013).
}
Functions contained in the package can be used to apply Algorithm 2 in 
Sevcikova et al. (2016) as shown in the Example below.
}
\examples{
# This example applies Algorithm 2 in Sevcikova et al. (2016) 
# on data from WPP2017 for China
#
data(mxM, mxF, e0Fproj, e0Mproj, package = "wpp2017")
country <- "China"

# extract observed mortality rates for male and female
mxm <- subset(mxM, name == country)[,4:16]
mxf <- subset(mxF, name == country)[,4:16]
rownames(mxm) <- rownames(mxf) <- c(0,1, seq(5, 100, by=5))

# Step 1: extrapolate from 100+ to 130+ using Coherent Kannisto
mx130 <- cokannisto(mxm, mxf)
 
# Steps 2-5: estimate coherent Lee-Carter parameters
# (here ax is computed from the last observed period 
# and smoothened over ages)
lc.est <- lileecarter.estimate(mx130$male, mx130$female, 
            ax.index = ncol(mx130$male), ax.smooth = TRUE)
        
# Steps 6-9: project future mortality rates based on future 
# life expectancies from WPP2017
e0f <- as.numeric(subset(e0Fproj, name == country)[-(1:2)])
e0m <- as.numeric(subset(e0Mproj, name == country)[-(1:2)])
names(e0f) <- names(e0m) <- colnames(e0Fproj)[-(1:2)]
pred <- mortcast(e0m, e0f, lc.est)

# plot projection for the first and last future time period
plot(pred$female$mx[,"2015-2020"], type="l", log="y", 
    ylim=range(pred$female$mx, pred$male$mx), xaxt="n",
    ylab="mx", xlab="Age", main=country, col="red")
axis(1, at=1:nrow(pred$female$mx),
    labels=rownames(pred$female$mx))
lines(pred$male$mx[,"2015-2020"], col="blue")
lines(pred$female$mx[,"2095-2100"], col="red", lty=2)
lines(pred$male$mx[,"2095-2100"], col="blue", lty=2)
legend("topleft", legend=c("male 2015-2020", "female 2015-2020",
    "male 2095-2100", "female 2095-2100"), bty="n",
    col=rep(c("blue", "red"),2), lty=c(1,1,2,2))

}
\references{
Li, N. and Lee, R. D. (2005). Coherent mortality forecasts for a group of populations: 
An extension of the Lee-Carter method. Demography, 42, 575-594.

Li, N., Lee, R. D. and Gerland, P. (2013). Extending the Lee-Carter method to model the rotation 
of age patterns of mortality decline for long-term projections. Demography, 50, 2037-2051.

Sevcikova H., Li N., Kantorova V., Gerland P., Raftery A.E. (2016). 
Age-Specific Mortality and Fertility Rates for Probabilistic Population Projections. 
In: Schoen R. (eds) Dynamic Demographic Analysis. The Springer Series on Demographic Methods
and Population Analysis, vol 39. Springer, Cham
}
