\name{model.mapmake}
\alias{model.mapmake}


\title{ Map Making }
\description{
  Applies models to either ERDAS Imagine image (.img) files or ESRI Grids of predictors to create detailed prediction surfaces.  It will handle large predictor files for map making, by reading in the \code{.img} files in rows, and output to the \code{.img} file the prediction for each data row, before reading the next row of data.
}
\usage{
model.mapmake(model.obj= NULL, folder = NULL, MODELfn = NULL, 
rastLUTfn = NULL, na.action = NULL, na.value=-9999, keep.predictor.brick=FALSE, 
map.sd = FALSE, OUTPUTfn = NULL, quantiles=NULL)
}

\arguments{
  \item{model.obj}{ \code{R} model object.  The model object to use for prediction.  The model object must be of type \code{"RF"} (random forest), \code{"QRF"} (quantile random forest), or \code{"CF"} (conditional forest). The \code{ModelMap} package does not currently support \code{SGB} models. }

  \item{folder}{ String.  The folder used for all output from predictions and/or maps.  Do not add ending slash to path string.  If \code{folder = NULL} (default), a GUI interface prompts user to browse to a folder.  To use the working directory, specify \code{folder = getwd()}.}

  \item{MODELfn}{ String.  The file name to use to save the generated model object.  If \code{MODELfn = NULL} (the default), a default name is generated by pasting \code{model.type_response.type_response.name}. If the other output filenames are left unspecified, \code{MODELfn} will be used as the basic name to generate other output filenames. The filename can be the full path, or it can be the simple basename, in which case the output will be to the folder specified by \code{folder}.}

  \item{rastLUTfn}{ String.  The file name (full path or base name with path specified by \code{folder}) of a \code{.csv} file for a \code{rastLUT}. Alternatively, a dataframe containing the same information. The \code{rastLUT} must include 3 columns: (1) the full path and name of the raster file; (2) the shortname of each predictor / raster layer (band); (3) the layer (band) number.  The shortname (column 2) must match the names \code{predList}, the predictor column names in training/test data set (\code{qdata.trainfn} and \code{qdata.testfn}, and the predictor names in \code{model.obj}. 

Example of comma-delimited file:

\tabular{llllll}{
	  \tab \tab \tab \code{C:/button_test/tc99_2727subset.img,} \tab \code{tc99_2727subsetb1,} \tab \code{1}\cr
	  \tab \tab \tab \code{C:/button_test/tc99_2727subset.img,} \tab \code{tc99_2727subsetb2,} \tab \code{2}\cr
	  \tab \tab \tab \code{C:/button_test/tc99_2727subset.img,} \tab \code{tc99_2727subsetb3,} \tab \code{3}}}

  \item{na.action}{String.  Model validation.  Specifies the action to take if there are \code{NA} values in the prediction data or if there is a level or class of a categorical predictor variable in the validation test set or the production (mapping) data set, but not in the training data set.  Currently, the only supported option for map making is \code{na.action = "na.omit"} (the default) where any data point or pixel with any new levels for any of the factored predictors is returned as \code{NA} (\code{na.value}, defaults to \code{-9999}).   }

  \item{na.value}{Number. Value that indicates \code{NA} in the predictor rasters. Defaults to \code{-9999}. Note: all predictor rasters must use the same value for \code{NA}. }

  \item{keep.predictor.brick}{Logical.  Map Production. If \code{TRUE} then the raster brick containing the predictors from the model object is saved as a native raster package format file. If \code{FALSE} a temporary brick is created and then deleted at the end of map production.   }

  \item{map.sd}{ Logical.  Map Production.  If \code{map.sd = TRUE}, maps of mean, standard deviation, and coefficient of variation of the predictions from all the trees are generated for each pixel.  If \code{map.sd = FALSE} (the default), only the predicted probability map will be built. 

This option is only available if the \code{model.type = "RF"} and the \code{response.type = "continuous"}.  Note: This option requires much more available memory.

The names of the additional maps default to: 

\tabular{llll}{
	  \tab \tab \tab \code{folder/model.type_response.type_response.name_mean.img}  \cr
	  \tab \tab \tab \code{folder/model.type_response.type_response.name_stdev.img}  \cr
	  \tab \tab \tab \code{folder/model.type_response.type_response.name_coefv.img} }
}



  \item{OUTPUTfn}{ String.  Map Production.  Filename of output file for map production.  The filename can be the full path, or it can be the simple basename, in which case the output will be to the folder specified by \code{folder}. If \code{OUTPUTfn = NULL} (the default), a name is created by pasting \code{modelfn} and \code{"_map"}. 

The \code{raster} package uses the filename extension to determine the output format of the map object. See  the \code{raster} package function \code{\link{writeFormats}} for a list of valid write formats. Because of this, only use the \code{"."} in output filenames if it is indicating a valid file extension.

If the output filename does not include an extension, the default extension of \code{".img"} will be added.

For continuous random forest models with \code{map.sd = TRUE} then \code{OUTPUTfn} is also used to create output file names for maps of the mean, standard deviation and coeficient of variation of all the trees predictions for each pixel.}

 \item{quantiles}{ Numeric Vector.  QRF models.  The quantiles to predict. A numeric vector with values between zero and one. The quantile map output will be a multilayer raster with one layer for each quantile. If \code{model.obj} was created with the \code{ModelMap} package, there will also be a single layer outputraster giving the ordinary RF mean predicted values. }

}
\details{

\code{model.mapmake()} can be run in a traditional R command mode, where all arguments are specified in the function call.  However it can also be used in a full push button mode, where you type in the simple command \code{model.mapmake()}, and GUI pop up windows will ask questions about the type of model, the file locations of the data, etc...

When running \code{model.mapmake()} on non-Windows platforms, file names and folders need to be specified in the argument list, but other pushbutton selections are handled by the \code{select.list()} function, which is platform independent. 

The R package \code{raster} is used to read spatial rasters into R. The data for production mapping should be in the form of pixel-based raster layers representing the predictors in the model. If there is more than one predictor or raster layer, the layers must all have the same number of columns and rows. The layers must also have the same extent, projection, and pixel size, for effective model development and accuracy. The \code{raster} package function \code{compareRaster()} is used to check predictor layers for consistency. 

The layers must also be in (single or multi-band) raster data formats that can be read by package \code{raster}, for example ESRI Grid or ERDAS Imagine image files. The predictor layers must have continuous or categorical data values. See \code{\link{writeRaster}} for a list of available formats.

To improve processing speed, the \code{raster} package is used to create a raster brick object with a layer for each predictor in the model. By default, this brick is a temporary file that is automatically deleated as soon as the map is completed. If \code{keep.predictor.brick=TRUE}, the predictor brick with be saved as a native \code{raster} package file, with a file name created by appending \code{'_brick'} to the \code{OUTPUTfn}. Warning: these bricks can be quite large, as they contain all the predictor data for every pixel in the map.

When creating maps of non-rectangular study regions there may be large portions of the rectangle where you have no predictors, and are uninterested in making predictions. The suggested value for the pixels outside the study area is \code{-9999}. These pixels will be ignored in the predictions, thus saving computing time. 

The function \code{model.mapmake()} outputs an rater file of map information suitable to be imported into a GIS. Maps can also be imported back into R using the function \code{raster()} from the \code{raster} package. The file extension of \code{OUTPUTfn} determines the write format. If \code{OUTPUTfn} does not include a file extension, output will default to an ERDAS Imagine image file with extension \code{".img"}

For Binary response models the output is in the form of predicted probability of presence for each pixel. For Continuous response models the output is the predicted value for each pixel. For Categorical response models the map output depends on the category labels. If the categorical response variable is numeric, the map output will use the original numeric categories. If the categories are non-numeric (for example, character strings), map output is in the form of integer class codes for each pixel, coded for each level of the factored response, and a CSV file containing a look up table is also generated to associate the integer codes with the original values of the response categories. 

The first predictor from \code{predList} is used to determine projection of output Imagine Image file.

}
\value{

The \code{model.mapmake()} function does not return a value, instead it writes a raster file of map information (suitable for importing into a GIS) to the specified folder. The output raster is saved in the format specifed by the file extension of \code{OUTPUTfn}

The \code{model.mapmake()} function also writes a text file listing the projections of all predictor rasters. 

For categorical response models, a csv file map key is written giving the integer code associated with each response category.

If \code{keep.predictor.brick = TRUE} then a raster brick of all the predictor rasters from the model is also saved to the specified folder. If \code{keep.predictor.brick = FALSE} (the default) then the predictor brick is written to a temprary file, and deleted. Warning: the predictor bricks can be quite large, and saving them can require quite a bit of memory.

}
\references{ 
Breiman, L. (2001) Random Forests. Machine Learning, 45:5-32.

Liaw, A. and  Wiener, M. (2002). Classification and Regression by randomForest. R News 2(3), 18--22.

Ridgeway, G., (1999). The state of boosting. Comp. Sci. Stat. 31:172-181

Simpson, E. H. (1949). Measurement of diversity. Nature.
 }


\note{If \code{model.mapmake()} is interupted it may leave orphan \code{.gri} and \code{.grd} files in your temporary directory. The \code{raster} package functions \code{showTmpFiles} and \code{removeTmpFiles} can be used to locate and remove these files, or they can be deleated manually from the temporary directory.
}

\author{ Elizabeth Freeman and Tracey Frescino }

\seealso{ \code{\link{get.test}}, \code{\link{model.build}}, \code{\link{model.diagnostics}}, \code{\link{compareRaster}}, \code{\link{writeRaster}}}
\examples{

\dontrun{

###########################################################################
############################# Run this set up code: #######################
###########################################################################

# set seed:
seed=38

# Define training and test files:

qdata.trainfn = system.file("extdata", "helpexamples","DATATRAIN.csv", package = "ModelMap")

# Define folder for all output:
folder=getwd()	


#identifier for individual training and test data points

unique.rowname="ID"


###########################################################################
######################## Define the model: ################################
###########################################################################


########## Continuous Response, Continuous Predictors ############

#file name to store model:
MODELfn="RF_Bio_TC"				

#predictors:
predList=c("TCB","TCG","TCW")	

#define which predictors are categorical:
predFactor=FALSE	

# Response name and type:
response.name="BIO"
response.type="continuous"



###########################################################################
########################### build model: ##################################
###########################################################################


### create model ###

model.obj = model.build( model.type="RF",
                       qdata.trainfn=qdata.trainfn,
                       folder=folder,		
                       unique.rowname=unique.rowname,		
                       MODELfn=MODELfn,
                       predList=predList,
                       predFactor=predFactor,
                       response.name=response.name,
                       response.type=response.type,
                       seed=seed,
                       na.action="na.roughfix"
)



###########################################################################
############ Then Run this code to predict map pixels #####################
###########################################################################


### Create a the filename (including path) for the rast Look up Tables ###


rastLUTfn.2001 <- system.file(	"extdata",
				"helpexamples",
				"LUT_2001.csv",
				package="ModelMap")
                         
                 



### Load rast LUT table, and add path to the predictor raster filenames in column 1 ###

rastLUT.2001 <- read.table(rastLUTfn.2001,header=FALSE,sep=",",stringsAsFactors=FALSE)

for(i in 1:nrow(rastLUT.2001)){
	rastLUT.2001[i,1] <- system.file("extdata",
					"helpexamples",
					rastLUT.2001[i,1],
					package="ModelMap")
}                                 


### Define filename for map  output ###

OUTPUTfn.2001 <- "RF_BIO_TCandNLCD_01.img"
OUTPUTfn.2001 <- paste(folder,OUTPUTfn.2001,sep="/")


### Create image files of predicted map data ###

model.mapmake( model.obj=model.obj,
               folder=folder,		
               rastLUTfn=rastLUT.2001,
           # Mapping arguments						
               OUTPUTfn=OUTPUTfn.2001
               )


###########################################################################
################ run this code to create maps in R ########################
###########################################################################

### Define Color Ramp ###

l <- seq(100,0,length.out=101)
c <- seq(0,100,length.out=101)
col.ramp <- hcl(h = 120, c = c, l = l)


### read in map data ###

mapgrid.2001 <- raster(OUTPUTfn.2001)

#mapgrid.2001 <- setMinMax(mapgrid.2001)


### create map ###

dev.new(width = 5, height = 5)
opar <- par(mar=c(3,3,2,1),oma=c(0,0,3,4),xpd=NA)

zlim <- c(0,max(maxValue(mapgrid.2001)))
legend.label<-rev(pretty(zlim,n=5))
legend.colors<-col.ramp[trunc((legend.label/max(legend.label))*100)+1]

image(  mapgrid.2001, col = col.ramp, zlim=zlim, asp=1, bty="n", 
        xaxt="n", yaxt="n", main="", xlab="", ylab="")
mtext("2001 Imagery",side=3,line=1,cex=1.2)

legend(	x=xmax(mapgrid.2001),y=ymax(mapgrid.2001),
	legend=legend.label,
	fill=legend.colors,
	bty="n",
	cex=1.2
)
mtext("Predictions",side=3,line=1,cex=1.5,outer=TRUE)
par(opar)

} # end dontrun

}

\keyword{ models }

