% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dateRange.R
\name{dateRange}
\alias{dateRange}
\title{Create a POSIXct date range}
\usage{
dateRange(startdate = NULL, enddate = NULL, timezone = NULL,
  unit = "sec", ceilingEnd = FALSE, days = 7)
}
\arguments{
\item{startdate}{Desired start datetime (ISO 8601).}

\item{enddate}{Desired end datetime (ISO 8601).}

\item{timezone}{Olson timezone used to interpret dates (required).}

\item{unit}{Units used to determine time at end-of-day.}

\item{ceilingEnd}{Logical instruction to apply
\code{\link[lubridate]{ceiling_date}} to the \code{enddate} rather than
\code{\link[lubridate]{floor_date}}}

\item{days}{Number of days of data to include.}
}
\value{
A vector of two \code{POSIXct}s.
}
\description{
Uses incoming parameters to return a pair of \code{POSIXct} times in the
proper order. The first returned time will be midnight of the desired
starting date. The second returned time will represent the "end of the day"
of the requested or calculated \code{enddate} boundary.

Note that the returned end date will be one \code{unit} prior to the start
of the requested \code{enddate} unless \code{ceilingEnd = TRUE} in
which case the entire \code{enddate} will be included up to the last
\code{unit}.

The \code{ceilingEnd} argument addresses the ambiguity of a phrase like:
"August 1-8". With \code{ceilingEnd = FALSE} (default) this pharse means
"through the beginning of Aug 8". With \code{ceilingEnd = TRUE} it means
"through the end of Aug 8".

So, to get 24 hours of data staring on Jan 01, 2019 you would specify:

\preformatted{
> MazamaCoreUtils::dateRange(20190101, 20190102, timezone = "UTC")
[1] "2019-01-01 00:00:00 UTC" "2019-01-01 23:59:59 UTC"
}

or

\preformatted{
> MazamaCoreUtils::dateRange(20190101, 20190101,
                             timezone = "UTC", ceilingEnd = TRUE)
[1] "2019-01-01 00:00:00 UTC" "2019-01-01 23:59:59 UTC"
}

The required \code{timezone} parameter must be one of those found in
\code{\link[base]{OlsonNames}}.

Dates can be anything that is understood by
\code{lubrdiate::parse_date_time()} using the \code{Ymd[HMS]} orders. This
includes:

\itemize{
  \item{\code{"YYYYmmdd"}}
  \item{\code{"YYYYmmddHHMMSS"}}
  \item{\code{"YYYY-mm-dd"}}
  \item{\code{"YYYY-mm-dd H"}}
  \item{\code{"YYYY-mm-dd H:M"}}
  \item{\code{"YYYY-mm-dd H:M:S"}}
}
}
\section{Default Arguments}{

In the case when either \code{startdate} or \code{enddate} is missing, it is
created from the non-missing values plus/minus \code{days}. If both
\code{startdate} and \code{enddate} are misssing, \code{enddate} is set to
\code{\link[lubridate]{now}} (with the given \code{timezone}), and then
\code{startdate} is calculated using \code{enddate - days}.
}

\section{End-of-Day Units}{

The second of the returned \code{POSIXct}s will end one \code{unit} before
the specified \code{enddate}. Acceptable units are \code{"day",
"hour", "min", "sec"}.

The aim is to quickly calculate full-day date ranges for time series whose
values are binned at different units. Thus, if \code{unit = "min"}, the
returned value associated with \code{enddate} will always be at 23:59:00
in the requested time zone.
}

\section{POSIXct inputs}{

When \code{startdate} or \code{enddate} are already \code{POSIXct} values,
they are converted to the timezone specified by \code{timezone} without
altering the physical instant in time the input represents. This is different
from the behavior of \code{\link[lubridate]{parse_date_time}} (which powers
this function), which will force \code{POSIXct} inputs into a new timezone,
altering the physical moment of time the input represents.
}

\section{Parameter precedence}{

It is possible to supply input paramters that are in conflict. For example:

\code{dateRange("2019-01-01", "2019-01-08", days = 3, timezone = "UTC")}

The \code{startdate} and \code{enddate} parameters would imply a 7-day range
which is in conflict with \code{days = 3}. The following rules resolve
conflicts of this nature:

\enumerate{
\item{When \code{startdate} and \code{enddate} are both specified, the
\code{days} parameter is ignored.}
\item{When \code{startdate} is missing, the first returned time will depend
on the combination of \code{enddate}, \code{days} and \code{ceilingEnd}.}
\item{When \code{enddate} is missing, \code{ceillingEnd} is ignored and the
second returned time depends on \code{days}.}
}
}

\examples{
dateRange("2019-01-08", timezone = "UTC")
dateRange("2019-01-08", unit = "min", timezone = "UTC")
dateRange("2019-01-08", unit = "hour", timezone = "UTC")
dateRange("2019-01-08", unit = "day", timezone = "UTC")
dateRange("2019-01-08", "2019-01-11", timezone = "UTC")
dateRange(enddate = 20190112, days = 3,
          unit = "day", timezone = "America/Los_Angeles")
}
