\name{mc.irf}
\alias{mc.irf}
\alias{mc.irf.VAR}
\alias{mc.irf.BVAR}
\alias{mc.irf.BSVAR}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{ Monte Carlo Integration / Simulation of Impulse Response
  Functions }
\description{
  Simulates a posterior sample of impulse response functions (IRF) by
  Monte carlo integration.  This can handle Bayesian and frequentist
  VARs and Bayesian structural VARs estimated with the szbvar(),
  szbsvar() or mlevar() functions.  The decomposition of the
  contemporaneous innovations is handled by a Cholesky decomposition of
  the error covariance matrix in each VAR object.  Simulations of IRFs
  from the Bayesian model utilize the posterior estimates for that model.
}
\usage{
mc.irf(varobj, nsteps, draws=0, A0.posterior=NULL)
mc.irf.VAR(varobj, nsteps, draws)
mc.irf.BVAR(varobj, nsteps, draws)
mc.irf.BSVAR(varobj, nsteps, A0.posterior)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{varobj}{ VAR objects for a fitted VAR model from either
    \code{szbvar()} or \code{reduced.form.var()}}
  \item{nsteps}{ Number of periods over which to compute the impulse responses}
  \item{draws}{ Number of draws for the simulation of the posterior
    distribution of the IRFs }
  \item{A0.posterior}{ Posterior sample objects generated by
    \code{gibbs.A0()}, based on the structural identification in
    \code{varobj$ident}}
}
\details{
  VAR/BVAR:
  
  Draws a set of posterior samples from the VAR coefficients and
  computes impulse responses for each sample.  These samples can then be
  summarized to compute MCMC-based estimates of the responses using the
  error band methods described in Sims and Zha (1999).

  B-SVAR:
  Generates a set of N2 draws from the impulse responses for the
  Bayesian SVAR model in \code{varobj}.  The function takes as its
  arguments the posterior moments of the B-SVAR model in \code{varobj}
  and the draws of the contemporaneous structural coefficients
  \eqn{A_0}{A(0)} from \code{gibbs.A0}. This function
  then computes a posterior sample of the impulse responses based on the
  draws of \eqn{A_0}{A(0)} and draws from the normal posterior pdf for
  the other coefficients in the model.

}
\value{
  VAR/BVAR:
  
  An mc.irf.VAR/mc.irf.BVAR class object object that is the array of
  impulse response samples for the Monte Carlo samples

  \item{impulse}{\eqn{draws \times nsteps \times m^2}{ draws X nsteps X
      (m*m)} array of the impulse responses }

  B-SVAR:
  mc.irf.BSVAR object which is an \eqn{(N2, nsteps, m^2)} array of the impulse
  responses for the associated B-SVAR model in \code{varobj} and the posterior
  \eqn{A_0}{A(0)}.
  
}

\references{
  Brandt, Patrick T. and John R. Freeman. 2006. "Advances in Bayesian
  Time Series Modeling and the Study of Politics: Theory Testing,
  Forecasting, and Policy Analysis"
   \emph{Political Analysis} 14(1):1-36.
   
  Sims, C.A. and Tao Zha. 1999. "Error Bands for Impulse
  Responses." \emph{Econometrica} 67(5): 1113-1156.

  Hamilton, James. 1994.  Time Series Analysis.  Chapter 11.

  Waggoner, Daniel F. and Tao A. Zha.  2003. "A Gibbs sampler for
  structural vector autoregressions" \emph{Journal of Economic Dynamics
  \& Control}. 28:349--366.

}
\author{ Patrick T. Brandt }
\note{ Users need to think carefully about the number of steps and the
  size of the posterior sample in \eqn{A_0}{A(0)}, since enough memory
  needs to be available to store and process the posterior of the
  impulse responses.   The amount of memory consumed by the impulse
  responses will be approximately \eqn{m^2 \times nsteps \times N2
    \times 16}{m^2 x nsteps x N2 x 16} where \code{N2} is the number of
  draws of \eqn{A_0}{A(0)} from the \code{gibbs.A0}.  Be sure
  you have enough memory available to store the object you create! }

\seealso{ See also as \code{\link{plot.mc.irf}} for plotting methods
  and error band construction for the posterior of the impulse response
  functions, \code{\link{szbsvar}} for estimation of the
  posterior moments of the B-SVAR model,  \code{\link{gibbs.A0}}
  for drawing posterior samples of \eqn{A_0}{A(0)} for the B-SVAR model
  before the IRF computations, and \code{\link{plot.mc.irf}} for a
  plotting method for the posterior of the impulse responses.
}
\examples{
  \dontrun{
    # Example 1
    data(IsraelPalestineConflict)
    varnames <- colnames(IsraelPalestineConflict)

    fit.BVAR <- szbvar(IsraelPalestineConflict, p=6, z=NULL,
                           lambda0=0.6, lambda1=0.1,
                            lambda3=2, lambda4=0.25, lambda5=0, mu5=0,
                            mu6=0, nu=3, qm=4,
                            prior=0, posterior.fit=FALSE)

    # Draw from the posterior pdf of the impulse responses.
    posterior.impulses <- mc.irf(fit.BVAR, nsteps=10, draws=5000)

    # Plot the responses
    plot(posterior.impulses, method=c("Sims-Zha2"), component=1,
                     probs=c(0.16,0.84), varnames=varnames)
}
\dontrun{
# Example 2
varobj <- szbsvar(Y, p, z = NULL,
                            lambda0, lambda1, lambda3, lambda4, lambda5,
                            mu5, mu6, ident, qm = 4)
A0.posterior <- gibbs.A0(varobj, N1, N2)
impulse.sample <- mc.irf(varobj, A0.posterior, nsteps)
}

}
\keyword{ ts}
\keyword{ models}
\keyword{ regression }
