% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gaps.R
\name{gap_handler}
\alias{gap_handler}
\title{Fill implicit gaps in a light logger dataset}
\usage{
gap_handler(
  dataset,
  Datetime.colname = Datetime,
  epoch = "dominant.epoch",
  behavior = c("full_sequence", "regulars", "irregulars", "gaps"),
  full.days = FALSE
)
}
\arguments{
\item{dataset}{A light logger dataset. Needs to be a dataframe.}

\item{Datetime.colname}{The column that contains the datetime. Needs to be a
\code{POSIXct} and part of the dataset.}

\item{epoch}{The epoch to use for the gapless sequence. Can be either a
\code{lubridate::duration()} or a string. If it is a string, it needs to be
either '"dominant.epoch"' (the default) for a guess based on the data or a valid \code{lubridate::duration()} string, e.g., \code{"1 day"} or \code{"10 sec"}.}

\item{behavior}{The behavior of the join of the \code{dataset} with the \code{gapless} sequence. Can be one of \code{"full_sequence"} (the default), \code{"regulars"}, \code{"irregulars"}, or \code{"gaps"}. See @return for details.}

\item{full.days}{If \code{TRUE}, the gapless sequence will include the whole first and last day where there is data.}
}
\value{
A modified \code{tibble} similar to \code{dataset} but with handling of implicit gaps, depending on the \code{behavior} argument:
\itemize{
\item \code{"full_sequence"} adds timestamps to the \code{dataset} that are missing based on a full sequence of \code{Datetimes} (i.e., the gapless sequence). The \code{dataset} is this equal (no gaps) or greater in the number of rows than the input. One column is added. \code{is.implicit} indicates whether the row was added (\code{TRUE}) or not (\code{FALSE}). This helps differentiating measurement values from values that might be imputed later on.
\item \code{"regulars"} keeps only rows from the gapless sequence that have a matching row in the dataset. This can be interpreted as a row-reduced \code{dataset} with only regular timestamps according to the \code{epoch}. In case of no gaps this tibble has the same number of rows as the input.
\item \code{"irregulars"} keeps only rows from the \code{dataset} that do not follow the regular sequence of \code{Datetimes} according to the \code{epoch}. In case of no gaps this tibble has 0 rows.
\item \code{"gaps"} returns a \code{tibble} of all implicit gaps in the dataset. In case of no gaps this tibble has 0 rows.
}
}
\description{
Datasets from light loggers often have implicit gaps. These gaps are implicit in the sense that consecutive timestamps (\code{Datetimes}) might not follow a regular epoch/interval. This function fills these implicit gaps by creating a gapless sequence of \code{Datetimes} and joining it to the dataset. The gapless sequence is determined by the minimum and maximum \code{Datetime} in the dataset (per group) and an epoch. The epoch can either be guessed from the dataset or specified by the user. A sequence of gapless \code{Datetimes} can be created with the \code{\link[=gapless_Datetimes]{gapless_Datetimes()}} function, whereas the dominant epoch in the data can be checked with the \code{\link[=dominant_epoch]{dominant_epoch()}} function. The \code{behaviour} argument specifies how the data is combined. By default, the data is joined with a full join, which means that all rows from the gapless sequence are kept, even if there is no matching row in the dataset.
}
\examples{
dataset <-
tibble::tibble(Id = c("A", "A", "A", "B", "B", "B"),
              Datetime = lubridate::as_datetime(1) +
                         lubridate::days(c(0:2, 4, 6, 8)) +
                         lubridate::hours(c(0,12,rep(0,4)))) \%>\% 
dplyr::group_by(Id)
dataset
#assuming the epoch is 1 day, we can add implicit data to our dataset
dataset \%>\% gap_handler(epoch = "1 day")

#we can also check whether there are irregular Datetimes in our dataset
dataset \%>\% gap_handler(epoch = "1 day", behavior = "irregulars")

#to get to the gaps, we can use the "gaps" behavior
dataset \%>\% gap_handler(epoch = "1 day", behavior = "gaps")
 
#finally, we can also get just the regular Datetimes
dataset \%>\% gap_handler(epoch = "1 day", behavior = "regulars")
}
\seealso{
Other regularize: 
\code{\link{dominant_epoch}()},
\code{\link{gap_finder}()},
\code{\link{gapless_Datetimes}()}
}
\concept{regularize}
