#' Initiate a conversation with a Generative AI model using a prompt
#'
#' This function establishes a connection to a Generative AI model by providing essential
#' parameters. It generates a chat response based on the provided prompt and stores it in the
#' history list.
#'
#' @param model.parameter A character vector containing information about the Generative AI service provider,
#' corresponding model, version, API key, and proxy status.
#' @param temperature A numeric value. Higher values yield more creative responses,
#' while lower values result in more straightforward text.
#' @param history A list containing all messages exchanged between the user and the model in a chat session.
#' @param prompt A character string representing the query for text generation.
#'
#' @return If successful, the full chat history, including the most recent response, will be returned. If
#' the API response indicates an error, the function halts execution and provides an error message.
#'
#' @details Providing accurate and valid information for each parameter is crucial for successful
#' text generation by the Generative AI model. This function modifies the chat history. If any
#' parameter is incorrect, the function responds with an error message based on the API feedback.
#' To view all supported Generative AI models, use the function \code{\link{available.models}}.
#' Additionally, utilize \code{\link{chat.setup}} to set up a chat history. To retrieve the most recent
#' response generated by the model, use the function \code{\link{chat.recent}}.
#'
#' @seealso
#' \href{https://genai.gd.edu.kg/r/documentation/}{GenAI - R Documentation}
#'
#' @examples
#' \dontrun{
#'  # Get available models
#'  models = available.models()
#'
#'  # Connect to the model, replace API_KEY with your api key
#'  google.model = connect.genai("google",
#'                               models$google$model[1],
#'                               models$google$version[1],
#'                               "API_KEY",
#'                               FALSE)
#'
#'  # Connect to the model, replace API_KEY with your api key
#'  openai.model = connect.genai("openai",
#'                               models$openai$model[1],
#'                               models$openai$version[1],
#'                               "API_KEY",
#'                               FALSE)
#'  # Setup an empty chat history
#'  google.history = chat.setup(google.model)
#'  openai.history = chat.setup(openai.model)
#'
#'  # Start chat
#'  temperature = 0.9
#'  prompt = "Write a story about a magic backpack in about 100 words."
#'  google.history = chat(google.model,
#'                        temperature,
#'                        google.history,
#'                        prompt)
#'
#'  # Get the most recent message generated by the model
#'  cat(chat.recent(google.model, google.history))
#'
#'  openai.history = chat(openai.model,
#'                        temperature,
#'                        openai.history,
#'                        prompt)
#'
#'  # Get the most recent message generated by the model
#'  cat(chat.recent(openai.model, openai.history))
#'
#'  prompt = "What is the word count of the story you just wrote?"
#'  google.history = chat(google.model,
#'                        temperature,
#'                        google.history,
#'                        prompt)
#'
#'  # Get the most recent message generated by the model
#'  cat(chat.recent(google.model, google.history))
#'
#'  openai.history = chat(openai.model,
#'                        temperature,
#'                        openai.history,
#'                        prompt)
#'
#'  # Get the most recent message generated by the model
#'  cat(chat.recent(openai.model, openai.history))
#' }
#'
#' @export
#'
#' @importFrom jsonlite toJSON
#' @importFrom httr POST add_headers content
chat = function(model.parameter,
                temperature,
                history,
                prompt) {
  if (prompt == "" ||
      is.na(prompt) || !inherits(prompt, "character")) {
    stop("Prompt is not in correct format.")
  }
  switch (model.parameter["provider"],
          google = {
            api.URL = ifelse(
              model.parameter["proxy"],
              paste0(
                "https://api.genai.gd.edu.kg/google/",
                model.parameter["version"],
                "/models/",
                model.parameter["model"],
                ":generateContent?key=",
                model.parameter["api"]
              ),
              paste0(
                "https://generativelanguage.googleapis.com/",
                model.parameter["version"],
                "/models/",
                model.parameter["model"],
                ":generateContent?key=",
                model.parameter["api"]
              )
            )
            requestNewContent = list(list(role = "user",
                                          parts = list(text = prompt)))
            config = list(generationConfig = list(temperature = temperature))
            requestBody = append(history, config)
            requestBody$contents = append(requestBody$contents, requestNewContent)
            requestBodyJSON = jsonlite::toJSON(requestBody, auto_unbox = TRUE)
            response = httr::POST(
              url = api.URL,
              body = requestBodyJSON,
              httr::add_headers("Content-Type" = "application/json")
            )
            responseJSON = httr::content(response, "parsed")
            if (!is.null(responseJSON$error)) {
              stop(responseJSON$error$message)
            }
            if (!is.null(responseJSON$blockReason)) {
              stop("The prompt may contain harmful content.")
            }
            history$contents = append(history$contents, requestNewContent)
            respondContent = list(list(
              role = "model",
              parts = list(text = responseJSON$candidates[[1]]$content$parts[[1]]$text)
            ))
            history$contents = append(history$contents, respondContent)
            return (history)
          },
          openai = {
            moderation.openai(model.parameter, prompt)
            api.URL = ifelse(
              model.parameter["proxy"],
              paste0(
                "https://api.genai.gd.edu.kg/openai/",
                model.parameter["version"],
                "/chat/completions"
              ),
              paste0(
                "https://api.openai.com/",
                model.parameter["version"],
                "/chat/completions"
              )
            )
            requestNewContent = list(list(role = "user",
                                          content = prompt))

            requestBody = append(history, list(temperature = temperature))
            requestBody$messages = append(requestBody$messages, requestNewContent)
            requestBodyJSON = jsonlite::toJSON(requestBody, auto_unbox = TRUE)
            response = httr::POST(
              url = api.URL,
              body = requestBodyJSON,
              httr::add_headers(
                "Content-Type" = "application/json",
                "Authorization" = paste("Bearer", model.parameter["api"])
              )
            )
            responseJSON = httr::content(response, "parsed")
            if (!is.null(responseJSON$error)) {
              stop(responseJSON$error$message)
            }
            history$messages = append(history$messages, requestNewContent)
            respondContent = list(list(
              role = "assistant",
              content = responseJSON$choices[[1]]$message$content
            ))
            history$messages = append(history$messages, respondContent)
            return (history)
          })
}
