% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Functions.R
\name{estimate_RWVAR_cp_heter}
\alias{estimate_RWVAR_cp_heter}
\title{Robust parameter estimation (RPE) for multivariate time series}
\usage{
estimate_RWVAR_cp_heter(
  data,
  L = 15,
  phiLower = -0.8,
  phiUpper = 0.8,
  sigetaLower = 0,
  sigetaUpper = Inf,
  signuLower = 1e-06,
  signuUpper = Inf,
  num_inis = 20,
  CPs = NULL
)
}
\arguments{
\item{data}{Numeric matrix of dimension \eqn{n \times p}, representing
the multivariate time series \eqn{\{\mathbf{y}_t\}_{t=1}^n}.}

\item{L}{Integer; number of lag differences used in each univariate RPE
estimation (default = 15).}

\item{phiLower, phiUpper}{Numeric; lower and upper bounds for the
autoregressive coefficient \eqn{\phi}.}

\item{sigetaLower, sigetaUpper}{Numeric; lower and upper bounds for
\eqn{\sigma_{\eta}^2}, the random walk innovation variance.}

\item{signuLower, signuUpper}{Numeric; lower and upper bounds for
\eqn{\sigma_{\nu}^2}, the VAR(1) innovation variance.}

\item{num_inis}{Integer; number of initial values of \eqn{\phi} used for
grid search initialization (default = 20).}

\item{CPs}{Optional numeric vector of change point locations (indices).
If provided, differenced data overlapping these points are removed for
more robust estimation.}
}
\value{
A list containing:
\itemize{
  \item `Sig_nu` — Diagonal matrix of estimated \eqn{\sigma_{\nu,i}^2}.
  \item `Sig_eta` — Diagonal matrix of estimated \eqn{\sigma_{\eta,i}^2}.
  \item `Phi` — Diagonal matrix of estimated autoregressive coefficients
    \eqn{\phi_i}.
}
}
\description{
Applies the robust parameter estimation (RPE) procedure componentwise
to a multivariate time series in order to estimate the diagonal elements
of \eqn{\Sigma_{\boldsymbol{\eta}}}, \eqn{\Sigma_{\boldsymbol{\nu}}}, and
\eqn{\Phi}.
}
\details{
This function performs the RPE procedure for each variable (column)
in `data` independently, using \code{\link{estimate_RWVAR_cp}} as the
univariate estimator. The resulting estimates are combined into
diagonal matrices:
\itemize{
  \item \eqn{\Sigma_{\boldsymbol{\nu}}} — estimated innovation covariance
    of the VAR(1) component.
  \item \eqn{\Sigma_{\boldsymbol{\eta}}} — estimated innovation covariance
    of the random walk component.
  \item \eqn{\Phi} — estimated autoregressive coefficient matrix.
}
}
\examples{
set.seed(123)
p <- 3

# True (diagonal) parameters for simulation
mu0    <- rep(0, p)
Sig_eta <- diag(0.01, p)
Sig_nu  <- random_Signu(p, 0)   # diagonal here since num_nonzero = 0
Phi     <- random_Phi(p, 0)     # diagonal here since num_nonzero = 0
Sig_e1  <- get_Sig_e1_approx(Sig_nu, Phi)

# Two evenly spaced change points
deltas <- list(c(3, 0, -3), c(-2, 4, 0))
Y <- generate_data(mu0, deltas, Sig_eta, Sig_nu, Phi, Sig_e1,
                   errortype = "n", number_cps = 2, lengthofeachpart = 100)

# Provide CP locations to remove affected differences in RPE
CPs <- c(100, 200)

# Componentwise robust parameter estimation
fit <- estimate_RWVAR_cp_heter(Y, L = 15, CPs = CPs)

# Estimated diagonal matrices:
fit$Sig_eta
fit$Sig_nu
fit$Phi

}
