% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kernel_fun.R
\name{kernel.DBR}
\alias{kernel.DBR}
\alias{kernel.DBR.fast}
\alias{kernel.DBR.crit}
\title{Kernel functions for DBR}
\usage{
kernel.DBR.fast(stepf, pv.numer, lambda)

kernel.DBR.crit(stepf, pv.numer, lambda, alpha, sorted.pv)
}
\arguments{
\item{stepf}{a list of the CDFs under the null hypothesis of each p-value.}

\item{pv.numer}{a numeric vector. Contains all values of the p-values supports if we search for the critical constants. If not, contains only the observed p-values. Must be in increasing order.}

\item{lambda}{a number strictly between 0 and 1. If \code{lambda=NULL} (by default), then \code{lambda} is chosen equal to \code{alpha}.}

\item{alpha}{the target FDR level, a number strictly between 0 and 1.}

\item{sorted.pv}{a vector of observed p-values, in increasing order.}
}
\value{
For \code{kernel.DBR.crit}, 
a list which elements are:
\item{crit.consts}{critical constants}
\item{pval.transf}{transformed p-values}
\item{m.lambda}{last index of observed p-values such that max_i F_i(p) <= lambda,
this needs to be passed to \code{\link{DBR}} to compute adjusted p-values).}

For \code{kernel.DBR.fast},
a vector of transformed p-values.
}
\description{
Kernel functions that transform 
observed p-values or their support according to [DBR-lambda].
The output is used by \code{\link{DBR}}.
Additionally, \code{kernel.DBR.crit} computes and returns
the critical constants.
The end user should not use them.
}
\details{
When computing critical constants, that is,
when using \code{kernel.DBR.crit}, we still need
to get transformed p-values to compute the adjusted 
p-values.

This version: 2018-11-13.
}
\examples{
X1 <- c(4, 2, 2, 14, 6, 9, 4, 0, 1)
X2 <- c(0, 0, 1, 3, 2, 1, 2, 2, 2)
N1 <- rep(148, 9)
N2 <- rep(132, 9)
Y1 <- N1-X1
Y2 <- N2-X2
df <- data.frame(X1,Y1,X2,Y2)
df
#Construction of the p-values and their support
df.formatted <- fisher.pvalues.support(counts = df, input = "noassoc")
raw.pvalues <- df.formatted$raw
pCDFlist <- df.formatted$support

m <- length(raw.pvalues)
alpha <- 0.05
lambda <- 0.05

#Compute the step functions from the supports
stepf <- build.stepfuns(pCDFlist)

#If not searching for critical constants, we use only the observed p-values
sorted.pvals <- sort(raw.pvalues)
y <- kernel.DBR.fast(stepf, sorted.pvals, lambda)

#If searching for critical constants, we use (almost) the complete support
pv.list.all <- unique(sort(as.numeric(unlist(pCDFlist))))
# apply the shortcut drawn from Corollary 3, that is
# c.1 >= the effective critical value associated to min((1 - lambda) * alpha/m , lambda)
pv.list <- short.eff(pv.list.all, min((1 - lambda) * alpha/m , lambda))
# then re-add the observed p-values (needed to compute the adjusted p-values),
# because we may have removed some of them the shortcut
pv.list <- sort(unique(c(pv.list, sorted.pvals)))
# compute transformed support
y <- kernel.DBR.crit(stepf, pv.list, lambda, alpha, sorted.pvals)
crit.constants <- y$crit.consts
transformed.pvalues <- y$pval.transf
last.index <- y$m.lambda

}
\seealso{
\code{\link{DBR}}, \code{\link{DiscreteFDR}},
\code{\link{kernel}}, \code{\link{discrete.BH}}
}
