#' A Reference Class to generates differents \code{calibrate.class} objects
#'
#' @description See the function \code{\link{calibrate}} which produces an instance of this class
#' This class comes with a set of methods, some of them being useful for the user:
#' See the documentation for \code{\link{calibrate}}... Other methods
#' should not be called as they are designed to be used during the calibration process.
#'
#' Fields should not be changed or manipulated by the user as they are updated internally
#' during the estimation process.
#' @field md a \code{\link{model.class}} object generated by \code{\link{model}}
#' @field pr a \code{\link{prior.class}} object generated by \code{\link{prior}}
#' @field opt.estim list of estimation options (see \code{\link{calibrate}} for details)
#' @field opt.valid list of cross validation options (see \code{\link{calibrate}} for details)
#' @field logPost the log posterior
#' @field mcmc a \code{coda} variable of the generated chains
#' @field output list of several chains and acceptation ratios
#' @field onlyCV activates only the cross validation
#' @field errorCV output of the CV errors
#' @field n.cores number cores available
#' @field binf the lower bound of the parameters for the DOE
#' @field bsup the upper bound of the parameters for the DOE
#' @export
calibrate.class <- R6Class(classname = "calibrate.class",
                           public = list(
                             md         = NULL,
                             pr         = NULL,
                             opt.estim  = NULL,
                             opt.valid  = NULL,
                             logPost    = NULL,
                             mcmc       = NULL,
                             output     = NULL,
                             onlyCV = NULL,
                             errorCV    = NULL,
                             ResultsCV  = NULL,
                             n.cores    = NULL,
                             binf       = NULL,
                             bsup       = NULL,
                             initialize = function(md=NA,pr=NA,opt.estim=NA,opt.valid=NULL,onlyCV)
                             {
                               #library(parallel)
                               self$onlyCV <- onlyCV
                               self$md         <- md
                               self$pr         <- pr
                               self$opt.estim  <- opt.estim
                               self$opt.valid  <- opt.valid
                               self$logPost    <- private$logLikelihood(self$md$model)
                               if (Sys.info()[['sysname']]=="Windows")
                               {
                                 self$n.cores    <- 1
                               } else
                               {
                                 self$n.cores    <- 2
                               }
                               if (self$opt.estim$burnIn > self$opt.estim$Nmh)
                               {
                                 stop("The burnIn must be inferior to Nmh")
                               }
                               if (self$onlyCV==FALSE)
                               {
                                 if (opt.estim$Nchains==1)
                                 {
                                   self$output  <- self$calibration()
                                   self$mcmc    <- as.mcmc(self$output$out$THETA)
                                 } else
                                 {
                                   n            <- c(1:opt.estim$Nchains)
                                   self$output  <- mclapply(n,self$calibration,mc.cores = self$n.cores)
                                   self$mcmc    <- list()
                                   for (i in 1:opt.estim$Nchains)
                                   {
                                     self$mcmc[[i]] <- as.mcmc(self$output[[i]]$out$THETA)
                                   }
                                 }
                                 cat("\nEnd of the regular calibration\n\n")
                               }
                               if (is.null(self$opt.valid)==FALSE)
                               {
                                 self$CV(1)
                                 cat(paste("\nThe cross validation is currently running on your ",
                                             self$n.cores," cores available....\n",sep=""))
                                 Results <- as.numeric(unlist(mclapply(c(1:opt.valid$nCV),
                                                                      self$CV,mc.cores = self$n.cores)))
                                 for (i in 1:self$opt.valid$nCV)
                                 {
                                   inc <- (i-1)*3+1
                                   if (i==1){
                                     self$ResultsCV <- data.frame(Predicted=Results[1],Real=Results[2],Error=Results[3])
                                     } else{
                                       self$ResultsCV <- rbind(self$ResultsCV,data.frame(Predicted=Results[inc],
                                                                                         Real=Results[inc+1],
                                                                                         Error=Results[inc+2]))
                                   }
                                 }
                                 self$errorCV <- sqrt(mean(self$ResultsCV$Error))
                               }
                             },
                             calibration = function(i=NA)
                             {
                               self$binf     <- private$boundaries()$binf
                               self$bsup     <- private$boundaries()$bsup
                               MetropolisCpp <- private$MCMC(self$md$model)
                               out           <- MetropolisCpp(self$opt.estim$Ngibbs,self$opt.estim$Nmh,
                                                              self$opt.estim$thetaInit,self$opt.estim$k,
                                                              self$opt.estim$sig,self$md$Yexp,self$binf,self$bsup,self$logPost,1)
                               MAP           <- private$MAPestimator(out)
                               return(list(out=out,MAP=MAP))
                             },
                             CV = function(i=NA)
                             {
                               if (self$opt.valid$type.valid=="loo")
                               {
                                 inc <- sample(c(1:self$md$n),1)
                                 if (is.matrix(self$md$X))
                                 {
                                   dataCal <- self$md$X[-inc,]
                                   dataVal <- as.matrix(t(self$md$X[inc,]))
                                 } else
                                 {
                                   dataCal <- self$md$X[-inc]
                                   dataVal <- self$md$X[inc]
                                 }
                                 Ycal        <- self$md$Yexp[-inc]
                                 Yval        <- self$md$Yexp[inc]
                                 mdTempCal   <- model(code=self$md$code,dataCal,Ycal,self$md$model,self$md$opt.emul)
                                 mdTempfit   <- self$calibrationCV(mdTempCal,Ycal)
                                 Dim         <- length(self$pr)
                                 thetaMAP    <- mdTempfit$MAP
                                 if (mdTempCal$model=="model1" | mdTempCal$model=="model2")
                                 {
                                    Predict  <- mdTempCal$pred(thetaMAP[1:(Dim-1)],thetaMAP[Dim],dataVal)$y
                                 } else
                                 {
                                    Predict  <- mdTempCal$pred(thetaMAP[1:(Dim-3)],thetaMAP[(Dim-2):(Dim-1)],
                                                               thetaMAP[Dim],dataVal)$y
                                 }
                                 err <- sqrt((Predict-Yval)^2)
                                 res <- list(Predict=Predict,Yval=Yval,err=err)
                                 return(res)
                               }
                             },
                             calibrationCV = function(mdTemp,y)
                             {
                               binf          <- private$boundaries()$binf
                               bsup          <- private$boundaries()$bsup
                               MetropolisCpp <- private$MCMC(mdTemp$model)
                               out           <- MetropolisCpp(self$opt.estim$Ngibbs,
                                                              self$opt.estim$Nmh,
                                                              self$opt.estim$thetaInit,self$opt.estim$k,
                                                              self$opt.estim$sig,y,binf,bsup,self$logPost,0)
                               MAP           <- private$MAPestimator(out)
                               return(list(out=out,MAP=MAP))
                             }
                           ))


calibrate.class$set("private","MCMC",
                  function(model)
                  {
                    switch(model,
                           model1={return(MetropolisHastingsCpp)},
                           model2={return(MetropolisHastingsCpp)},
                           model3={return(MetropolisHastingsCppD)},
                           model4={return(MetropolisHastingsCppD)}
                    )
                  })


calibrate.class$set("private","boundaries",
                function()
                {
                  binf <- self$pr[[1]]$binf
                  bsup <- self$pr[[1]]$bsup
                  for (i in 2:length(self$pr))
                  {
                    binf <- c(binf,self$pr[[i]]$binf)
                    bsup <- c(bsup,self$pr[[i]]$bsup)
                  }
                  return(list(binf=binf,bsup=bsup))
                })


calibrate.class$set("private","logLikelihood",
                function(model)
                {
                  switch(model,
                         model1={return(private$logTest)},
                         model2={return(private$logTest)},
                         model3={return(private$logTestD)},
                         model4={return(private$logTestD)}
                  )
                }
)

calibrate.class$set("private","logTest",
                    function(theta,sig2)
                    {
                      if (length(self$pr) == 1)
                      {
                        return(log(self$md$likelihood(theta,sig2))+self$pr$prior(theta))
                      } else
                      {
                        s <- 0
                        for (i in 1:(length(theta)))
                        {
                          s <- s + self$pr[[i]]$prior(theta[i])
                        }
                        s <- s + self$pr[[(length(theta)+1)]]$prior(sig2)
                        return(self$md$likelihood(theta,sig2) + s)
                      }
                    })

calibrate.class$set("private","logTestD",
                function(theta,thetaD,sig2)
                {
                  s <- 0
                  for (i in 1:(length(theta)))
                  {
                    s <- s + self$pr[[i]]$prior(theta[i])
                  }
                  for (j in 1:(length(thetaD)))
                  {
                    s <- s + self$pr[[length(theta)+j]]$prior(thetaD[j])
                  }
                  s <- s + self$pr[[(length(theta)+1)]]$prior(sig2)
                  return(self$md$likelihood(theta,thetaD,sig2) + s)
                })

calibrate.class$set("private","MAPestimator",
                    function(out)
                      {
                        chain <- out$THETA[-c(1:self$opt.estim$burnIn),]
                        dens <- apply(chain,2,density)
                        # if (chain[nrow(chain),1]==chain[1,1] & chain[nrow(chain),2]==chain[1,2])
                        # {
                        #   cat("\n")
                        #   cat("Call:\n")
                        #   print(self$md$model)
                        #   cat("\n")
                        #   cat("With the function:\n")
                        #   print(self$md$code)
                        #   cat("\n")
                        #   cat("Acceptation rate of the Metropolis within Gibbs algorithm:\n")
                        #   print(paste(round(out$AcceptationRatioWg/self$opt.estim$Ngibbs*100,2),
                        #               "%",sep = ""))
                        #   cat("\n")
                        #   cat("Acceptation rate of the Metropolis Hastings algorithm:\n")
                        #   print(paste(out$AcceptationRatio/self$opt.estim$Nmh*100,"%",sep = ""))
                        #   cat("\n")
                        #   stop('the MCMC chain did not move, please try to launch again with different values of k')
                        # }
                        map <- function(dens)
                        {
                          dens$x[which(dens$y==max(dens$y))]
                        }
                        return(unlist(lapply(dens,map)))
                    })


calibrate.class$set("public","plot",
                    function(graph=c("acf","chains","densities","output"),select.X=NULL)
                    {
                      if (self$onlyCV==TRUE)
                      {
                        stop("You only had requested the cross validation, the plot method is desabled")
                      }
                      if (self$opt.estim$Nchains>1)
                      {
                        warning("You have selected several chains. The plot is from coda package and the coda object
                                mcmc")
                        plot(self$mcmc)
                        stop("")
                      }
                      n   <- length(self$pr)
                      gg  <- list()
                      a   <- list()
                      m   <- list()
                      p   <- list()
                      if ("acf" %in% graph)
                      {
                        for (i in 1:n)
                        {
                          a[[i]] <- self$acf(i)
                        }
                        # do.call(grid.arrange,a)
                        gg$acf <- a
                      }
                      if ("chains" %in% graph)
                      {
                        for (i in 1:n)
                        {
                          m[[i]] <- self$mcmcChains(i)
                        }
                        # do.call(grid.arrange,m)
                        gg$mcmc <- m
                      }
                      if ("densities" %in% graph)
                      {
                        for (i in 1:n)
                        {
                          dplot2  <- data.frame(data=self$output$out$THETA[-c(1:self$opt.estim$burnIn),i],
                                                type="posterior")
                          p[[i]] <- self$pr[[i]]$plot()+geom_density(data=dplot2,kernel="gaussian",adjust=3,alpha=0.1)
                        }
                        # do.call(grid.arrange,p)
                        gg$dens <- p
                      }
                      if ("output" %in% graph)
                      {
                        o <- self$outputPlot(select.X)
                        # print(o)
                        gg$output <- o
                      }
                      return(gg)
                    })


calibrate.class$set("public","acf",
                    function(i)
                    {
                      bacf   <- acf(self$output$out$THETA[-c(1:self$opt.estim$burnIn),i], plot = FALSE)
                      bacfdf <- with(bacf, data.frame(lag, acf))
                      p      <- ggplot(data = bacfdf, mapping = aes(x = lag, y = acf))+
                        geom_hline(aes(yintercept = 0))+
                        geom_segment(mapping = aes(xend = lag, yend = 0))+
                        xlab("")+ylab("")+theme_light()
                      return(p)
                      })

calibrate.class$set("public","mcmcChains",
                    function(i)
                    {
                      n <- length(self$output$out$THETA[-c(1:self$opt.estim$burnIn),i])
                      resgg <- data.frame(inc=c(1:n),data=self$output$out$THETA[-c(1:self$opt.estim$burnIn),i])
                      p   <- ggplot(data=resgg, aes(x=inc,y=data))+geom_line()+ylab("")+
                        xlab("")+theme_light()
                      return(p)
                    }
                    )


calibrate.class$set("public","outputPlot",
                    function(select.X=NULL)
                    {
                      if (is.null(select.X)==TRUE)
                        {
                          if (is.null(dim(self$md$X))==TRUE)
                          {
                            X <- self$md$X
                          }else
                          {
                            stop('The dimension of X is higher than 1, the plot cannot be provided for a dimension >1')
                          }
                        } else {X <- select.X}
                      m <- self$output$out$THETA[-c(1:self$opt.estim$burnIn),]
                      Dist <- Dist2 <- matrix(nr=nrow(m),nc=length(self$md$Yexp))
                      dim   <- length(self$pr)
                      if (self$md$model == "model1" || self$md$model == "model2")
                      {
                        parFun <- function(i)
                        {
                          D  <- self$md$fun(m[i,1:(dim-1)],m[i,dim])$y
                          return(D)
                        }
                        res <- mclapply(1:nrow(m),parFun,mc.cores = self$n.cores)
                        for (i in 1:nrow(m))
                        {
                          Dist[i,] <- res[[i]]
                        }
                      } else
                      {
                        print('The computational time might be long to get the output plot')
                        parFun <- function(i)
                        {
                          D  <- self$md$fun(m[i,1:(dim-3)],m[i,(dim-2):(dim-1)],m[i,dim])$y
                          D2 <- self$md$fun(self$output$MAP[1:(dim-3)],m[i,(dim-2):(dim-1)],
                                                   self$output$MAP[dim])$y
                          return(list(D=D,D2=D2))
                        }
                        res <- mclapply(1:nrow(m),parFun,mc.cores = self$n.cores)
                        for (i in 1:nrow(m))
                        {
                          Dist[i,]  <- res[[i]]$D
                          Dist2[i,] <- res[[i]]$D2
                        }
                        qqd <- apply(Dist2,2,quantile,probs=c(0.05,0.95))
                        ggdata2 <- data.frame(y=self$md$Yexp,x=X,upper=qqd[2,],lower=qqd[1,],type="experiments",
                                            fill="90% credibility interval for the discrepancy")
                      }
                      qq <- apply(Dist,2,quantile,probs=c(0.05,0.95))
                      if (self$md$model=="model1"||self$md$model=="model2")
                      {
                        MAP <- self$output$MAP
                        p <- length(MAP)-1
                        Ys <- self$md$fun(MAP[1:p],MAP[p+1])$y
                      }else
                      {
                        MAP <- self$output$MAP
                        p <- length(MAP)-3
                        Ys <- self$md$fun(MAP[1:p],MAP[(p+1):(p+2)],MAP[p+3])$y
                      }
                      ggdata <- data.frame(y=Ys,x=X,upper=qq[2,],lower=qq[1,],type="calibrated",
                                           fill="90% credibility interval a posteriori")
                      if (self$md$model == "model1" || self$md$model == "model2")
                      {
                        ggdata2 <- data.frame(y=self$md$Yexp,x=X,upper=qq[2,],lower=qq[1,],type="experiments",
                                              fill="90% credibility interval a posteriori")
                      }
                      ggdata <- rbind(ggdata,ggdata2)
                      p <- ggplot(ggdata) + geom_line(aes(x=x,y=y,color=type))+
                        geom_ribbon(aes(ymin=lower, ymax=upper, x=x,fill=fill), alpha = 0.3) +
                        scale_fill_manual("",values=c("blue4","black")) +
                        theme_light() + xlab("") + ylab("") +
                        theme(legend.position=c(0.65,0.86),
                              legend.text=element_text(size = '15'),
                              legend.title=element_blank(),
                              legend.key=element_rect(colour=NA),
                              axis.text=element_text(size=20))
                     return(p)
                    })


calibrate.class$set("public","print",
                    function()
                    {
                      cat("Call:\n\nWith the function:\n")
                      print(self$md$code)
                      cat("\nSelected model :",self$md$model,"\n")
                      if (self$onlyCV==TRUE)
                      {
                        cat("\nCross validation:\n Method:",self$opt.valid$type.valid,"\n")
                        print(head(self$ResultsCV))
                        cat("\nRMSE: ")
                        print(self$errorCV)
                      } else
                      {
                        cat("\nAcceptation rate of the Metropolis within Gibbs algorithm:\n")
                        print(paste(round(self$output$out$AcceptationRatioWg/self$opt.estim$Ngibbs*100,2),
                                    "%",sep = ""))
                        cat("\nAcceptation rate of the Metropolis Hastings algorithm:\n")
                        print(paste(self$output$out$AcceptationRatio/self$opt.estim$Nmh*100,"%",sep = ""))
                        cat("\nMaximum a posteriori:\n")
                        print(apply(self$output$out$THETA[-c(1:self$opt.estim$burnIn),],2,max))
                        cat("\nMean a posteriori:\n")
                        print(apply(self$output$out$THETA[-c(1:self$opt.estim$burnIn),],2,mean))
                        cat("\n")
                        if (is.null(self$opt.valid)==FALSE)
                        {
                          cat("\nCross validation:\n Method:",self$opt.valid$type.valid,"\n")
                          print(head(self$ResultsCV))
                          cat("\nRMSE: ")
                          print(self$errorCV)
                        }
                      }
                    }
)

