#' Preference Learning with the Mallows Rank Model
#'
#' Compute the posterior distributions of the parameters of the Bayesian Mallows
#' Rank Model \insertCite{mallows1957,vitelli2018}{BayesMallows}, given rankings
#' or preferences stated by a set of assessors. \code{compute_mallows} always
#' returns posterior distributions of the latent consensus ranking \eqn{\rho}
#' and the scale parameter \eqn{\alpha}. Several distance measures are
#' supported, and the preferences can take the form of complete or incomplete
#' rankings, as well as pairwise preferences. \code{compute_mallows} can also
#' compute mixtures of Mallows models, for clustering of assessors with similar
#' preferences.
#'
#'
#' @param rankings A matrix of ranked items, of size \code{n_assessors x
#'   n_items}. See \code{\link{create_ranking}} if you have an ordered set of
#'   items that need to be converted to rankings. If \code{preferences} is
#'   provided, \code{rankings} is an optional initial value of the rankings,
#'   generated by \code{\link{generate_initial_ranking}}. If \code{rankings} has
#'   column names, these are assumed to be the names of the items.
#'
#' @param preferences A dataframe with pairwise comparisons, with 3 columns,
#'   named \code{assessor}, \code{bottom_item}, and \code{top_item}, and one row
#'   for each stated preference. Given a set of pairwise preferences, generate a
#'   transitive closure using \code{\link{generate_transitive_closure}}. This
#'   will give \code{preferences} the class \code{"BayesMallowsTC"}. If
#'   \code{preferences} is not of class \code{"BayesMallowsTC"},
#'   \code{compute_mallows} will call \code{\link{generate_transitive_closure}}
#'   on \code{preferences} before computations are done. In the current version,
#'   the pairwise preferences are assumed to be mutually compatible.
#'
#' @param metric A character string specifying the distance metric to use in the
#'   Bayesian Mallows Model. Available options are \code{"footrule"},
#'   \code{"spearman"}, \code{"kendall"}, and \code{"cayley"}. The distance
#'   given by \code{metric} is also used to compute within-cluster distances,
#'   when \code{include_wcd = TRUE}.
#'
#' @param n_clusters Integer specifying the number of clusters, i.e., the number
#'   of mixture components to use. Defaults to \code{1L}, which means no
#'   clustering is performed. See \code{\link{compute_mallows_mixtures}} for a
#'   convenience function for computing several models with varying numbers of
#'   mixtures.
#'
#' @param cluster_assignment_thinning Integer specifying the thinning to be
#'   applied to the cluster assignments. Defaults to \code{1L}.
#'
#' @param nmc Integer specifying the number of iteration of the
#'   Metropolis-Hastings algorithm. Defaults to \code{2000L}. See
#'   \code{\link{assess_convergence}} for tools to check convergence of the
#'   Markov chain.
#'
#' @param leap_size Integer specifying the step size of the leap-and-shift
#'   proposal distribution. Defaults \code{floor(n_items / 5)}.
#'
#' @param rho_init Numeric vector specifying the initial value of the latent
#'   consensus ranking \eqn{\rho}. Defaults to NULL, which means that the
#'   initial value is set randomly. If \code{rho_init} is provided when
#'   \code{n_clusters > 1}, each mixture component \eqn{\rho_{c}} gets the same
#'   initial value.
#'
#' @param rho_thinning Integer specifying the thinning of \code{rho} to be
#'   performed in the Metropolis- Hastings algorithm. Defaults to \code{1L}.
#'   \code{compute_mallows} save every \code{rho_thinning}th value of
#'   \eqn{\rho}. See \insertCite{link2011;textual}{BayesMallows} for a
#'   discussion of when it is appropriate to use thinning, and when it is not.
#'
#' @param alpha_prop_sd Numeric value specifying the standard deviation of the
#'   lognormal proposal distribution used for \eqn{\alpha} in the
#'   Metropolis-Hastings algorithm. Defaults to \code{0.1}.
#'
#' @param alpha_init Numeric value specifying the initial value of the scale
#'   parameter \eqn{\alpha}. Defaults to \code{1}. When \code{n_clusters > 1},
#'   each mixture component \eqn{\alpha_{c}} gets the same initial value.
#'
#' @param alpha_jump Integer specifying how many times to sample \eqn{\rho}
#'   between each sampling of \eqn{\alpha}. In other words, how many times to
#'   jump over \eqn{\alpha} while sampling \eqn{\rho}, and possibly other
#'   parameters like augmented ranks \eqn{\tilde{R}} or cluster assignments
#'   \eqn{z}. Setting \code{alpha_jump} to a high number can speed up
#'   computation time, by reducing the number of times the partition function
#'   for the Mallows model needs to be computed.
#'
#' @param lambda Numeric value specifying the rate parameter of the exponential
#'   prior distribution of \eqn{\alpha}, \eqn{\pi(\alpha) = \lambda
#'   \exp{(-\lambda \alpha)}}. Defaults to \code{0.1}. When \code{n_cluster >
#'   1}, each mixture component \eqn{\alpha_{c}} has the same prior
#'   distribution.
#'
#' @param psi Integer specifying the concentration parameter \eqn{\psi} of the
#'   Dirichlet prior distribution used for the cluster probabilities
#'   \eqn{\tau_{1}, \tau_{2}, \dots, \tau_{C}}, where \eqn{C} is the value of
#'   \code{n_clusters}. Defaults to \code{10L}. When \code{n_clusters = 1}, this
#'   argument is not used.
#'
#' @param include_wcd Logical indicating whether to store the within-cluster
#'   distances computing during the Metropolis-Hastings algorithm. Defaults to
#'   \code{TRUE} if \code{n_clusters > 1} and otherwise \code{FALSE}. Setting
#'   \code{include_wcd = TRUE} is useful when deciding the number of mixture
#'   components to include, and is required by \code{\link{plot_elbow}}.
#'
#' @param save_augmented_data Logical specifying whether or not to save the
#'   augmented rankings every \code{aug_thinning}th iteration, for the case of
#'   missing data or pairwise preferences. Defaults to \code{FALSE}. Saving
#'   augmented data is useful for predicting the rankings each assessor would
#'   give to the items not yet ranked, and is required by
#'   \code{\link{plot_top_k}}.
#'
#' @param aug_thinning Integer specifying the thinning for saving augmented
#'   data. Only used when \code{save_augmented_data = TRUE}. Defaults to 1L.
#'
#' @param logz_estimate Estimate of the partition function, computed with
#'   \code{\link{estimate_partition_function}}. Be aware that when using an
#'   estimated partition function when \code{n_clusters > 1}, the partition
#'   function should be estimated over the whole range of \eqn{\alpha} values
#'   covered by the prior distribution for \eqn{\alpha} with high probability.
#'   In the case that a cluster \eqn{\alpha_c} becomes empty during the
#'   Metropolis-Hastings algorihm, the posterior of \eqn{\alpha_c} equals its
#'   prior. For example, if the rate parameter of the exponential prior equals,
#'   say \eqn{\lambda = 0.001}, there is about 37 \% (or exactly: \code{1 -
#'   pexp(1000, 0.001)}) prior probability that \eqn{\alpha_c > 1000}. Hence
#'   when \code{n_clusters > 1}, the estimated partition function should cover
#'   this range, or \eqn{\lambda} should be increased.
#'
#' @param verbose Logical specifying whether to print out the progress of the
#'   Metropolis-Hastings algorithm. If \code{TRUE}, a notification is printed
#'   every 1000th iteration.
#'
#' @param validate_rankings Logical specifying whether the rankings provided (or
#'   generated from \code{preferences}) should be validated. Defaults to
#'   \code{TRUE}. Turning off this check will reduce computing time with a large
#'   number of items or assessors.
#'
#' @param constraints Optional constraint set returned from
#'   \code{\link{generate_constraints}}. Defaults to \code{NULL}, which means
#'   the the constraint set is computed internally. In repeated calls to
#'   \code{compute_mallows}, with very large datasets, computing the constraint
#'   set may be time consuming. In this case it can be beneficial to precompute
#'   it and provide it as a separate argument.
#'
#' @param skip_postprocessing Logical specifying whether to skip the
#'   postprocessing of the output of the Metropolis-Hastings algorithm. This can
#'   be useful for very large datasets, which cause the postprocessing to crash.
#'   Note that when \code{skip_postprocessing=TRUE}, the functions for studying
#'   the posterior distributions will not work unless the internal function
#'   \code{\link{tidy_mcmc}} has been run.
#'
#' @return A list of class BayesMallows.
#'
#' @seealso \code{\link{compute_mallows_mixtures}} for a function that computes
#'   separate Mallows models for varying number of clusters.
#'
#'
#'
#' @references \insertAllCited{}
#'
#' @export
#' @importFrom rlang .data
#'
#' @example /inst/examples/compute_mallows_example.R
#'
compute_mallows <- function(rankings = NULL,
                            preferences = NULL,
                            metric = "footrule",
                            n_clusters = 1L,
                            cluster_assignment_thinning = 1L,
                            nmc = 2000L,
                            leap_size = floor(n_items / 5),
                            rho_init = NULL,
                            rho_thinning = 1L,
                            alpha_prop_sd = 0.1,
                            alpha_init = 1,
                            alpha_jump = 1L,
                            lambda = 0.001,
                            psi = 10L,
                            include_wcd = (n_clusters > 1),
                            save_augmented_data = FALSE,
                            aug_thinning = 1L,
                            logz_estimate = NULL,
                            verbose = FALSE,
                            validate_rankings = TRUE,
                            constraints = NULL,
                            skip_postprocessing = FALSE
                            ){

  # Check that at most one of rankings and preferences is set
  stopifnot(!is.null(rankings) || !is.null(preferences))

  if(!is.null(rho_init)) {
    stopifnot(validate_permutation(rho_init) && (sum(is.na(rho_init)) == 0))
  }

  # Deal with pairwise comparisons. Generate rankings compatible with them.
  if(!is.null(preferences)){
    if(!("BayesMallowsTC" %in% class(preferences))){
      message("Generating transitive closure of preferences.")
      preferences <- generate_transitive_closure(preferences)
    }
    if(is.null(rankings)){
      message("Generating initial ranking.")
      rankings <- generate_initial_ranking(preferences)
    }
  }

  # Check that all rows of rankings are proper permutations
  if(validate_rankings && !all(apply(rankings, 1, validate_permutation))){
    stop("Not valid permutation.")
  }

  # Check that we do not jump over all alphas
  stopifnot(alpha_jump < nmc)

  # Check that we do not jump over all rhos
  stopifnot(rho_thinning < nmc && aug_thinning < nmc)

  # Find the number of items
  n_items <- ncol(rankings)


  # Generate the constraint set
  if(!is.null(preferences) && is.null(constraints)){
    constraints <- generate_constraints(preferences, n_items)
  } else if (is.null(constraints)){
    constraints <- list()
  }


  # Set leap_size if it is not alredy set.
  #if(is.null(leap_size)) leap_size <- floor(n_items / 5)

  # Extract the right sequence of cardinalities, if relevant
  if(!is.null(logz_estimate)){
    cardinalities <- NULL
    message("Using user-provided importance sampling estimate of partition function.")
  } else if(metric %in% c("footrule", "spearman")){
    # Extract the relevant rows from partition_function_data
    # Note that we need to evaluate the right-hand side, in particular metric,
    # to avoid confusion with columns of the tibble
    relevant_params <- dplyr::filter(partition_function_data,
                                     .data$n_items == !!n_items,
                                     .data$metric == !!metric
    )

    type <- dplyr::pull(relevant_params, type)

    if((length(type) == 0) || !(type %in% c("cardinalities", "importance_sampling"))){
      stop("Precomputed partition function not available yet. Consider computing one
           with the function estimate_partition_function(), and provide it
           in the logz_estimate argument to compute_mallows().")
    } else {
      cardinalities <- unlist(relevant_params$values)
      logz_estimate <- NULL
    }

  } else if (metric %in% c("cayley", "kendall")) {
    cardinalities <- NULL
    logz_estimate <- NULL
  } else {
    stop(paste("Unknown metric", metric))
  }


  # Transpose rankings to get samples along columns, since we typically want
  # to extract one sample at a time. armadillo is column major, just like rankings
  fit <- run_mcmc(rankings = t(rankings),
                  nmc = nmc,
                  constraints = constraints,
                  cardinalities = cardinalities,
                  logz_estimate = logz_estimate,
                  rho_init = rho_init,
                  metric = metric,
                  n_clusters = n_clusters,
                  include_wcd = include_wcd,
                  lambda = lambda,
                  leap_size = leap_size,
                  alpha_prop_sd = alpha_prop_sd,
                  alpha_init = alpha_init,
                  alpha_jump = alpha_jump,
                  rho_thinning = rho_thinning,
                  aug_thinning = aug_thinning,
                  cluster_assignment_thinning = cluster_assignment_thinning,
                  save_augmented_data = save_augmented_data,
                  verbose = verbose
                  )

  if(verbose){
    print("Metropolis-Hastings algorithm completed. Post-processing data.")
  }

  # Add some arguments
  fit$metric <- metric
  fit$lambda <- lambda
  fit$nmc <- nmc
  fit$n_items <- n_items
  fit$n_clusters <- n_clusters
  fit$alpha_jump <- alpha_jump
  fit$rho_thinning <- rho_thinning
  fit$aug_thinning <- aug_thinning
  fit$leap_size <- leap_size
  fit$alpha_prop_sd <- alpha_prop_sd
  fit$include_wcd <- include_wcd
  fit$save_augmented_data <- save_augmented_data

  # Add names of item
  if(!is.null(colnames(rankings))) {
    fit$items <- colnames(rankings)
  } else {
    fit$items <- paste("Item", seq(from = 1, to = nrow(fit$rho), by = 1))
  }

  # Tidy MCMC results
  if(!skip_postprocessing) fit <- tidy_mcmc(fit)

  # Add class attribute
  class(fit) <- "BayesMallows"

  return(fit)

}
