% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{cox_bvs}
\alias{cox_bvs}
\title{Non-parallel version of Bayesian variable selector for survival data using
nonlocal priors}
\usage{
cox_bvs(exmat, cur_cols, nf, tau, r, nlptype, a, b, d, L, J, temps)
}
\arguments{
\item{exmat}{An extended matrix where the first two columns are survival
times and status, respectively and the rest is the design matrix which is
produced by \code{PreProcess} function.}

\item{cur_cols}{A vector containing indices of the initial model for
variable selector to start the S5 algorithm from. Note that the first
\code{nf} indices are \code{1} to \code{nf} where \code{nf} is the number
of fixed covariates that do not enter the selection procedure.}

\item{nf}{The number of fixed covariates that do not enter the selection
procedure.}

\item{tau}{The paramter \code{tau} of the iMOM prior.}

\item{r}{The paramter \code{r} of the iMOM prior.}

\item{nlptype}{Determines the type of nonlocal prior that is used in the
analyses. \code{0} is for piMOM and \code{1} is for pMOM.}

\item{a}{The first parameter in beta distribution used as prior on model
size. This parameter is equal to 1 when uinform-binomial prior is used.}

\item{b}{The second paramter in beta distribution used as prior on model
size. This parameter is equal to 1 when uinform-binomial prior is used.}

\item{d}{This is the number of candidate covariates picked from top
variables with highest utility function value and used in S5 algorithm.}

\item{L}{Number of temperatures in S5 algorithm.}

\item{J}{Number of iterations at each temperature in S5 algorithm.}

\item{temps}{Vector of temperatuers used in S5 algorithm.}
}
\value{
It returns a list containing following objects:
\item{max_model}{A \code{1} by \code{p} binary vector showing the selected
model with maximum probability. \code{1} means a specific variable is
selected.}
\item{hash_key}{A column vector indicating the generated key for each model
that is used to track visited models and growing dictionary.}
\item{max_prob}{The unnormalized probability of the model with highest
posterior probability.}
\item{all_probs}{A vector containing unnormalized probabilities of all
visited models.}
\item{vis_covs_list}{A list containing the covariates in each visited model
in the stochastic search process.}
}
\description{
This function performs Bayesian variable selection for
survival data in a non-parallel fashion. It runs modified S5 algorithm to
search the model space but since this is only on one CPU, the number of
visited models will not be large and therefore is NOT recommended for
high dimensional datasets. This function is called by \code{\link{bvs}}
function in a parllel fashion and therefore that function is recommended
to be used.
}
\examples{
### Initializing the parameters
n <- 100
p <- 40
set.seed(123)
Sigma <- diag(p)
full <- matrix(c(rep(0.5, p*p)), ncol=p)
Sigma <- full + 0.5*Sigma
cholS <- chol(Sigma)
Beta <- c(-1.8, 1.2, -1.7, 1.4, -1.4, 1.3)
X = matrix(rnorm(n*p), ncol=p)
X = X\%*\%cholS
X <- scale(X)
beta <- numeric(p)
beta[c(1:length(Beta))] <- Beta
XB <- X\%*\%beta
sur_times <- rexp(n,exp(XB))
cens_times <- rexp(n,0.2)
times <- pmin(sur_times,cens_times)
status <- as.numeric(sur_times <= cens_times)
exmat <- cbind(times,status,X)
L <- 10; J <- 10
d <- 2 * ceiling(log(p))
temps <- seq(3, 1, length.out = L)
tau <- 0.5; r <- 1; a <- 6; b <- p-a
nlptype <- 0 ### PiMOM nonlocal prior
cur_cols <- c(1,2,3) ### Starting model for the search algorithm
nf <- 0 ### No fixed columns

### Running the Function
coxout <- cox_bvs(exmat,cur_cols,nf,tau,r,nlptype,a,b,d,L,J,temps)

### The number of visited model for this specific run:
length(coxout$hash_key)


### The selected model:
which(coxout$max_model>0)

### The unnormalized probability of the selected model:
coxout$max_prob

}
\references{
Nikooienejad, A., Wang, W., and Johnson, V. E. (2017). Bayesian
Variable Selection in High Dimensional Survival Time Cancer Genomic
Datasets using Nonlocal Priors. arXiv preprint, arXiv:1712.02964.\cr\cr
Shin, M., Bhattacharya, A., and Johnson, V. E. (2017). Scalable
Bayesian variable selection using nonlocal prior densities in ultrahigh
dimensional settings. Statistica Sinica.\cr\cr
Johnson, V. E., and Rossell, D. (2010). On the use of non-local prior
densities in Bayesian hypothesis tests. Journal of the Royal Statistical
Society: Series B (Statistical Methodology), 72(2), 143-170.
}
\seealso{
\code{\link{bvs}}
}
\author{
Amir Nikooienejad
}
