#' The bivariate Wrapped Normal distribution
#' @inheritParams rvmsin
#' @inheritParams rwnorm
#' @param mu1,mu2 vectors of mean parameters.
#' @param kappa1,kappa2,kappa3 vectors of concentration parameters; \code{kappa1, kappa2 > 0},
#' and \code{kappa3^2 < kappa1*kappa2}.
#' @details
#' The bivariate wrapped normal density at the point \eqn{x = (x_1, x_2)} is given by,
#' \deqn{f(x) = \sqrt((\kappa_1 \kappa_2 - (\kappa_3)^2) / (2\pi)) \sum \exp(-1/2 * (\kappa_1 (T_1)^2 + \kappa_2 (T_2)^2 + \kappa_3 (T_1) (T_2)) )}
#' where
#' \deqn{T_1 = T_1(x, \mu, \omega) = (x_1 - \mu_1(2\pi\omega_1))}
#' \deqn{T_2 = T_2(x, \mu, \omega) = (x_2 - \mu_1(2\pi\omega_2))}
#' the sum extends over all pairs of integers \eqn{\omega = (\omega_1, \omega_2)},
#' and is approximated by a sum over \eqn{(\omega_1, \omega_2)} in \eqn{\{-M, -M+1, ..., M-1, M \}^2} if \code{int.displ = } \eqn{M}.
#' @return \code{dwnorm2} gives the density  and \code{rwnorm2} generates random deviates.
#'
#' @examples
#' kappa1 <- c(1, 2, 3)
#' kappa2 <- c(1, 6, 5)
#' kappa3 <- c(0, 1, 2)
#' mu1 <- c(1, 2, 5)
#' mu2 <- c(0, 1, 3)
#' x <- diag(2, 2)
#' n <- 10
#'
#' # when x is a bivariate vector and parameters are all scalars,
#' # dwnorm2 returns single density
#' dwnorm2(x[1, ], kappa1[1], kappa2[1], kappa3[1], mu1[1], mu2[1])
#'
#' # when x is a two column matrix and parameters are all scalars,
#' # dmvsin returns a vector of densities calculated at the rows of
#' # x with the same parameters
#' dwnorm2(x, kappa1[1], kappa2[1], kappa3[1], mu1[1], mu2[1])
#'
#' # if x is a bivariate vector and at least one of the parameters is
#' # a vector, all parameters are recycled to the same length, and
#' # dwnorm2 returns a vector of with ith element being the density
#' # evaluated at x with parameter values kappa1[i], kappa2[i],
#' # kappa3[i], mu1[i] and mu2[i]
#' dwnorm2(x[1, ], kappa1, kappa2, kappa3, mu1, mu2)
#'
#' # if x is a two column matrix and at least one of the parameters is
#' # a vector, rows of x and the parameters are recycled to the same
#' # length, and dwnorm2 returns a vector of with ith element being the
#' # density evaluated at ith row of x with parameter values kappa1[i],
#' # kappa2[i], # kappa3[i], mu1[i] and mu2[i]
#' dwnorm2(x[1, ], kappa1, kappa2, kappa3, mu1, mu2)
#'
#' # when parameters are all scalars, number of observations generated
#' # by rwnorm2 is n
#' rwnorm2(n, kappa1[1], kappa2[1], kappa3[1], mu1[1], mu2[1])
#'
#' # when at least one of the parameters is a vector, all parameters are
#' # recycled to the same length, n is ignored, and the number of
#' # observations generated by rwnorm2 is the same as the length of the
#' # recycled vectors
#' rwnorm2(n, kappa1, kappa2, kappa3, mu1, mu2)
#'
#' @export

rwnorm2 <- function(n, kappa1=1, kappa2=1, kappa3=0, mu1=0, mu2=0)
{
  if(any(c(kappa1, kappa2) <= 0)) stop("kappa1 and kappa2 must be positive")
  if(any(mu1 < 0 | mu1 >= 2*pi)) mu1 <- prncp_reg(mu1)
  if(any(mu2 < 0 | mu2 >= 2*pi)) mu2 <- prncp_reg(mu2)

  if(max(length(kappa1), length(kappa2), length(kappa3), length(mu1), length(mu2)) > 1) {
    expanded <- expand_args(kappa1, kappa2, kappa3, mu1, mu2)
    kappa1 <- expanded[[1]]; kappa2 <- expanded[[2]]; kappa3 <- expanded[[3]]
    mu1 <- expanded[[4]]; mu2 <- expanded[[5]]
    if(any(kappa1*kappa2 - kappa3*kappa3 <= 1e-10))
      stop("abs(kappa3) must be less than sqrt(kappa1*kappa2) in wnorm2")
    m <- length(kappa1)
    mu_list <- lapply(1:m, function(j) c(mu1[j], mu2[j]))
    sigma_list <- lapply(1:m, function(j) matrix(c(kappa2[j], -kappa3[j], -kappa3[j],
                                                   kappa1[j])/(kappa1[j]*kappa2[j] - kappa3[j]^2), 2))
    samp <- t(sapply(1:m, function(j) rnorm2(1, mu_list[[j]], sigma_list[[j]])))

  } else {
    if(kappa1*kappa2 - kappa3*kappa3 <= 1e-10)
      stop("abs(kappa3) must be less than sqrt(kappa1*kappa2) in wnorm2")
    mu <- c(mu1, mu2)
    sigma <- matrix(c(kappa2, -kappa3, -kappa3, kappa1)/(kappa1*kappa2 - kappa3*kappa3), 2)
    samp <- rnorm2(n, mu, sigma)
  }
  prncp_reg(samp)
}

#' @rdname rwnorm2
#' @export

dwnorm2 <- function(x, kappa1=1, kappa2=1, kappa3=0, mu1=0, mu2=0, int.displ)
{
  if(missing(int.displ)) int.displ <- 3
  else if(int.displ >= 5) int.displ <- 5
  else if(int.displ <= 1) int.displ <- 1
  displ <- floor(int.displ)
  omega.2pi.all <- expand.grid(-displ:displ,-displ:displ) * (2*pi) # 2pi * integer displacements
  omega.2pi <- as.matrix(omega.2pi.all)

  if(any(c(kappa1, kappa2) <= 0)) stop("kappa1 and kappa2 must be positive")
  if(any(mu1 < 0 | mu1 >= 2*pi)) mu1 <- prncp_reg(mu1)
  if(any(mu2 < 0 | mu2 >= 2*pi)) mu2 <- prncp_reg(mu2)
  if((length(dim(x)) < 2 && length(x) != 2) || (length(dim(x)) == 2 && tail(dim(x), 1) != 2)
     || (length(dim(x)) > 2)) stop("x must either be a bivariate vector or a two-column matrix")

  if(max(length(kappa1), length(kappa2), length(kappa3), length(mu1), length(mu2)) > 1) {
    expanded <- expand_args(kappa1, kappa2, kappa3, mu1, mu2)
    kappa1 <- expanded[[1]]; kappa2 <- expanded[[2]]; kappa3 <- expanded[[3]]
    mu1 <- expanded[[4]]; mu2 <- expanded[[5]]
    if(any(kappa1*kappa2 - kappa3^2 <= 1e-10))
      stop("abs(kappa3) must be less than sqrt(kappa1*kappa2) in wnorm2")
    par.mat <- rbind(kappa1,kappa2,kappa3,mu1,mu2)
    if(length(x) != 2) {
      x_set <- 1:nrow(x)
      par_set <- 1:length(kappa1)
      expndn_set <- expand_args(x_set, par_set)
      x_set <- expndn_set[[1]]
      par_set <- expndn_set[[2]]
      as.vector(dwnorm2_manyx_manypar(x[x_set, ], kappa1[par_set], kappa2[par_set], kappa3[par_set], mu1[par_set], mu2[par_set], omega.2pi))
    } else{
      as.vector(dwnorm2_onex_manypar(x, kappa1, kappa2, kappa3, mu1, mu2, omega.2pi))
    }
  } else {
    if(kappa1*kappa2 - kappa3^2 <= 1e-10)
      stop("abs(kappa3) must be less than sqrt(kappa1*kappa2) in wnorm2")
    if(length(x) != 2){
      as.vector(dwnorm2_manyx_onepar(x, kappa1, kappa2, kappa3, mu1, mu2, omega.2pi))
    } else{
      l.const.wnorm2 <- l_const_wnorm2(c(kappa1, kappa2, kappa3))
      exp(ldwnorm2_num(x, c(kappa1, kappa2, kappa3, mu1, mu2), omega.2pi) - l.const.wnorm2)
    }
  }
}

#' The bivariate Wrapped Normal mixtures
#' @inheritParams rvmsinmix
#' @inheritParams rwnorm
#' @param mu1,mu2 vectors of mean parameters.
#' @param kappa1,kappa2,kappa3 vectors of concentration parameters; \code{kappa1, kappa2 > 0, kappa3^2 < kappa1*kappa2} for each component.
#'
#' @details All the argument vectors \code{pmix, kappa1, kappa2, kappa3, mu1} and \code{mu2} must be of the same length,
#' with \eqn{j}-th element corresponding to the \eqn{j}-th component of the mixture distribution.
#' @details The bivariate wrapped normal mixture distribution with component size \code{K = \link{length}(pmix)} has density
#' \deqn{g(x) = \sum p[j] * f(x; \kappa_1[j], \kappa_2[j], \kappa_3[j], \mu_1[j], \mu_2[j])}
#' where the sum extends over \eqn{j}; \eqn{p[j]; \kappa_1[j], \kappa_2[j], \kappa_3[j]}; and \eqn{\mu_1[j], \mu_2[j]} respectively denote the mixing proportion,
#' the three concentration parameters and the two mean parameter for the \eqn{j}-th component, \eqn{j = 1, ..., K},
#' and \eqn{f(. ; \kappa_1, \kappa_2, \kappa_3, \mu_1, \mu_2)} denotes the density function of the wrapped normal distribution
#' with concentration parameters \eqn{\kappa_1, \kappa_2, \kappa_3} and  mean parameters \eqn{\mu_1, \mu_2}.
#' @return \code{dwnorm2mix} computes the density  and \code{rwnorm2mix} generates random deviates from the mixture density.
#'
#' @examples
#' kappa1 <- c(1, 2, 3)
#' kappa2 <- c(1, 6, 5)
#' kappa3 <- c(0, 1, 2)
#' mu1 <- c(1, 2, 5)
#' mu2 <- c(0, 1, 3)
#' pmix <- c(0.3, 0.4, 0.3)
#' x <- diag(2, 2)
#' n <- 10
#'
#' # mixture densities calculated at the rows of x
#' dwnorm2mix(x, kappa1, kappa2, kappa3, mu1, mu2, pmix)
#'
#' # number of observations generated from the mixture distribution is n
#' rwnorm2mix(n, kappa1, kappa2, kappa3, mu1, mu2, pmix)
#'
#' @export

rwnorm2mix <- function(n, kappa1, kappa2, kappa3, mu1, mu2, pmix)
{
  allpar <- list(kappa1=kappa1, kappa2=kappa2, kappa3=kappa3,
                 mu1=mu1, mu2=mu2, pmix=pmix)

  allpar_len <- listLen(allpar)
  if(min(allpar_len) != max(allpar_len))
    stop("component size mismatch: number of components of the input parameter vectors differ")

  if(any(allpar$pmix < 0)) stop("\'pmix\' must be non-negative")
  sum_pmix <- sum(allpar$pmix)
  if(signif(sum_pmix, 5) != 1) {
    if(sum_pmix <= 0) stop("\'pmix\' must have at least one positive element")
    allpar$pmix <- allpar$pmix/sum_pmix
    warning("\'pmix\' is rescaled to add up to 1")
  }

  if(any(c(allpar$kappa1, allpar$kappa2) <= 0)) stop("kappa1 and kappa2 must be positive")
  if(any(allpar$kappa1*allpar$kappa2 - allpar$kappa3^2 <= 1e-10))
    stop("abs(kappa3) must be less than sqrt(kappa1*kappa2) in wnorm2")
  if(any(allpar$mu1 < 0 | allpar$mu1 >= 2*pi)) allpar$mu1 <- prncp_reg(allpar$mu1)
  if(any(allpar$mu2 < 0 | allpar$mu2 >= 2*pi)) allpar$mu2 <- prncp_reg(allpar$mu2)

  clus_label <- cID(t(replicate(allpar$pmix, n = n)), length(allpar$pmix), runif(n))
  rwnorm2(1, allpar$kappa1[clus_label], allpar$kappa2[clus_label], allpar$kappa3[clus_label],
          allpar$mu1[clus_label], allpar$mu2[clus_label])
}


#' @rdname rwnorm2mix
#' @export
dwnorm2mix <- function(x, kappa1, kappa2, kappa3, mu1, mu2, pmix, int.displ)
{
  if(missing(int.displ)) int.displ <- 3
  else if(int.displ >= 5) int.displ <- 5
  else if(int.displ <= 1) int.displ <- 1
  displ <- floor(int.displ)
  omega.2pi.all <- expand.grid(-displ:displ,-displ:displ) * (2*pi) # 2pi * integer displacements
  omega.2pi <- as.matrix(omega.2pi.all)

  allpar <- list("kappa1"=kappa1, "kappa2"=kappa2, "kappa3"=kappa3,
                 "mu1"=mu1, "mu2"=mu2, "pmix"=pmix)

  allpar_len <- listLen(allpar)
  if(min(allpar_len) != max(allpar_len)) stop("component size mismatch: number of components of the input parameter vectors differ")

  if(any(allpar$pmix < 0)) stop("\'pmix\' must be non-negative")
  sum_pmix <- sum(allpar$pmix)
  if(signif(sum_pmix, 5) != 1) {
    if(sum_pmix <= 0) stop("\'pmix\' must have at least one positive element")
    allpar$pmix <- allpar$pmix/sum_pmix
    warning("\'pmix\' is rescaled to add up to 1")
  }

  if(any(allpar$kappa1*allpar$kappa2 - allpar$kappa3*allpar$kappa3 <= 1e-10))
    stop("abs(kappa3) must be less than sqrt(kappa1*kappa2) in wnorm2")

  if(any(c(allpar$kappa1, allpar$kappa2) <= 0)) stop("kappa1 and kappa2 must be positive")
  if(any(allpar$mu1 < 0 | allpar$mu1 >= 2*pi)) allpar$mu1 <- prncp_reg(allpar$mu1)
  if(any(allpar$mu2 < 0 | allpar$mu2 >= 2*pi)) allpar$mu2 <- prncp_reg(allpar$mu2)
  if((length(dim(x)) < 2 && length(x) != 2) || (length(dim(x)) == 2 && tail(dim(x), 1) != 2)
     || (length(dim(x)) > 2)) stop("x must either be a bivariate vector or a two-column matrix")


  par_mat <- rbind(allpar$kappa1, allpar$kappa2, allpar$kappa3, allpar$mu1, allpar$mu2)
  pi_mix <- allpar$pmix
  log_c_von = log_const_wnorm2_all(par_mat)

  if(length(x) == 2) {
    wnorm2mix(x, par_mat, pi_mix, log_c_von, omega.2pi)
  } else {
    as.vector(wnorm2mix_manyx(x, par_mat, pi_mix, log_c_von, omega.2pi))
  }
}


#' Fitting bivariate wrapped normal mixtures using MCMC
#' @inheritParams fit_vmsinmix
#' @param int.displ absolute integer displacement for each coordinate. Default is 3.
#' Allowed minimum and maximum are 1 and 5 respectively.
#' @param epsilon,L  tuning parameters for HMC; ignored if \code{method = "rwmh"}. \code{epsilon} (step-size) is a quantity in
#' \eqn{[0, 1)} and \code{L} (leapfrog steps) is a positive integer.
#' @param gam.loc,gam.scale location and scale (hyper-) parameters for the gamma prior for \code{kappa1} and \code{kappa2}. See
#' \link{dgamma}. Defaults are \code{gam.loc = 0, gam.scale = 1000} that makes the prior non-informative.
#'
#' @details
#' \code{fit_wnorm2mix} generates MCMC samples of wnorm2 mixture model parameters, and returns an
#' angmcmc object as the output, which can be used as an argument for diagnostics and estimation
#' functions.
#'
#' Default \code{method} is \code{"hmc"}.
#'
#' If the acceptance rate drops below 5\% after 100 or more HMC iterations, \code{epsilon} is automatically lowered, and the
#' Markov chain is restarted at the current parameter values.
#'
#' @usage
#' fit_wnorm2mix(data, ncomp, start_par = list(), method = "hmc",
#'               epsilon = 0.005, L = 10, epsilon.random = TRUE,
#'               L.random = FALSE, propscale = rep(0.01, 5),
#'               n.iter = 500, int.displ, gam.loc = 0,
#'               gam.scale = 1000, norm.var = 1000, autotune = FALSE,
#'               iter.tune = 10, ncores, show.progress = TRUE)
#'
#' @return returns an angmcmc object.
#'
#' @examples
#' # illustration only - more iterations needed for convergence
#' fit.wnorm2.15 <- fit_wnorm2mix(tim8, ncomp = 3, n.iter =  15,
#'                                ncores = 1)
#' fit.wnorm2.15
#'
#' @export

fit_wnorm2mix <- function(data, ncomp, start_par = list(), method="hmc", epsilon=0.005, L=10, epsilon.random=TRUE,
                          L.random=FALSE, propscale = rep(0.01, 5), n.iter=500, int.displ, gam.loc=0, gam.scale=1000,
                          norm.var=1000, autotune = FALSE, iter.tune=10, ncores, show.progress = TRUE) {

  if(is.null(dim(data)) | !(mode(data) %in% c("list", "numeric") && ncol(data) == 2)) stop("non-compatible data")

  if(missing(int.displ)) int.displ <- 3
  else if(int.displ >= 5) int.displ <- 5
  else if(int.displ <= 1) int.displ <- 1

  displ <- floor(int.displ)
  omega.2pi.all <- expand.grid(-displ:displ,-displ:displ) * (2*pi) # 2pi * integer displacements
  omega.2pi <- as.matrix(omega.2pi.all)


  kappa_upper <- 150
  curr.model <- "wnorm2"
  data.rad <- rm_NA_rad(data)
  n.data <- nrow(data.rad)

  if(missing(ncores)) {
    ncores <- floor(parallel::detectCores())
  }

  if(ncomp == 1) {

    if(missing(start_par)) {
      starting <- start_clus_kmeans_wnorm2(data.rad, ncomp, nstart=5)
      starting$par.mat <- matrix(starting$par.mat, ncol=1)
    } else {
      allpar <- start_par
      if(any(is.null(allpar$kappa1), is.null(allpar$kappa2), is.null(allpar$kappa3),
             is.null(allpar$mu1), is.null(allpar$mu2)) ) {
        stop("too few elements in start_par, with no default")
      }
      allpar1 <- list(allpar$kappa1, allpar$kappa2, allpar$kappa3, allpar$mu1, allpar$mu2)
      allpar_len <- listLen(allpar1)
      if(min(allpar_len) != max(allpar_len)){
        stop("component size mismatch: number of components of the input parameter vectors differ")
      }
      if(any((allpar$kappa1)*(allpar$kappa2) - (allpar$kappa3)^2 <= 1e-10)) {
        stop("abs(kappa3) must be less than sqrt(kappa1*kappa2) in wnorm2")
      }
      starting <- list("par.mat" = rbind(start_par$kappa1, start_par$kappa2, start_par$kappa3,
                                         start_par$mu1, start_par$mu2), "pi.mix" = 1)
    }
  } else if(ncomp > 1) {
    if(missing(start_par)) {
      starting <- start_clus_kmeans_wnorm2(data.rad, ncomp, nstart=5)
    } else {
      allpar <- start_par
      if(any(is.null(allpar$kappa1), is.null(allpar$kappa2), is.null(allpar$kappa3),
             is.null(allpar$mu1), is.null(allpar$mu2), is.null(allpar$pmix)) ) {
        stop("too few elements in start_par, with no default")
      }
      allpar1 <- list(allpar$kappa1, allpar$kappa2, allpar$kappa3, allpar$mu1, allpar$mu2, allpar$pmix)
      allpar_len <- listLen(allpar1)
      if(min(allpar_len) != max(allpar_len)){
        stop("component size mismatch: number of components of the input parameter vectors differ")
      }
      if(any((allpar$kappa1)*(allpar$kappa2) - (allpar$kappa3)^2 <= 1e-10)) {
        stop("abs(kappa3) must be less than sqrt(kappa1*kappa2) in wnorm2")
      }
      starting <- list("par.mat" = rbind(start_par$kappa1, start_par$kappa2, start_par$kappa3,
                                         start_par$mu1, start_par$mu2), "pi.mix" = start_par$pmix)
    }
  }

  starting$par.mat[abs(starting$par.mat) >= kappa_upper/2] <- kappa_upper/2
  starting$l.c.wnorm2 <- as.numeric(log_const_wnorm2_all(starting$par.mat))
  starting$llik <- llik_wnorm2_full(data.rad, starting$par.mat, starting$pi.mix, starting$l.c.wnorm2, omega.2pi, ncores)
  starting$lprior <- sum(ldgamanum(starting$par.mat[1:2,], gam.loc, gam.scale)) - 0.5*sum((starting$par.mat[3,]/norm.var)^2)
  starting$lpd <- starting$llik + starting$lprior

  par.mat.all <- array(0, dim = c(5, ncomp, n.iter+1))
  pi.mix.all <- matrix(1, nrow = ncomp, ncol = n.iter+1)
  llik.all <- lprior.all <- lpd.all <- 1:(n.iter+1)
  accpt.par.mat.all <- accpt.kappa.all <- accpt.mu.all <- rep(0, (n.iter+1))
  modelpar.names <- c("kappa1", "kappa2", "kappa3", "mu1", "mu2")

  MC <- starting  #simulation results list, 1st entry = method of moments on kmeans output

  par.mat.all[,,1] <- MC$par.mat
  pi.mix.all[,1] <- MC$pi.mix
  llik.all[1] <- MC$llik
  lprior.all[1] <- MC$lprior
  lpd.all[1] <- MC$lpd

  epsilon_ave <- NULL
  L_ave <- NULL
  propscale_final <- NULL

  clus.ind <- matrix(1, nrow = n.data, ncol = n.iter+1) # will be replaced if ncomp > 1

  iter <- 2
  ntune <- 0

  if(show.progress) pb <- txtProgressBar(min = 2, max = n.iter+1, style = 3)

  #******************************************************************************************
  # single component model
  #******************************************************************************************

  if(ncomp == 1 && grepl(method, "hmc")) # using hmc
  {
    if(epsilon.random)
      epsilon_vec <- runif(n.iter, min = 0.9*epsilon, max = 1.1*epsilon)
    if(L.random)
      L_vec <- sample(1:L, n.iter, replace = TRUE)


    while(iter <= (n.iter+1)) {
      broken <- FALSE
      kappa.large <- FALSE
      pi.mix.1 <- 1
      par.mat.old <- MC$par.mat
      l.c.wnorm2.old <- MC$l.c.wnorm2
      llik_new.pi <- MC$llik

      #----------------------------------------------------------------------------------
      #generating par.mat by HMC
      #----------------------------------------------------------------------------------

      par.mat.1 <- par.mat.old
      lprior.1 <- MC$lprior
      llik.1 <- llik_new.pi
      lpd.1 <- llik.1 + lprior.1
      l.c.wnorm2.1 <- MC$l.c.wnorm2
      accpt.par.mat <- 0


      current_q <- par.mat.1
      current_p <- matrix(rnorm(5*ncomp,0,1), nrow = 5)  # independent standard normal variates
      p <- current_p
      q <- current_q

      if(L.random)
        L <- L_vec[iter-1]

      if(epsilon.random)
        epsilon <- epsilon_vec[iter-1]

      # Do leapfrog with L and epsilon
      {
        # Make a half step for momentum at the beginning

        p <- p - (epsilon/2) * (- grad_wnorm2_all_comp(data.rad, q, pi.mix.1, omega.2pi, ncores)
                                + matrix(c(1/gam.scale + (1- 1/gam.scale)/q[1:2,], q[3,], rep(0,2)), ncol=1)
        ) # the second term in the bracket arises from prior
        # Alternate full steps for position and momentum

        for (i in 1:L)
        {
          # Make a full step for the position

          q <- q + epsilon * p

          if(all(!is.nan(q)) && any(abs(q[1:3, ]) >= kappa_upper)) {
            kappa.large <- TRUE
            break
          }

          if(any(is.nan(c(q, p)))) {
            broken <- TRUE
            #stop("Algorithm breaks. Try a smaller epsilon.")
            break
          }
          # Make sure the components of q1 are in the proper ranges
          {
            q1 <- q; p1 <- p

            for(j in 1:ncomp) {

              while(q1[1,j] <= 0) {
                q1[1,j] <- -q1[1,j]; p1[1,j] <- -p1[1,j]
              }

              while(q1[2,j] <= 0) {
                q1[2,j] <- -q1[2,j]; p1[2,j] <- -p1[2,j]
              }

              q_3.upper <- sqrt(q1[1,j]*q1[2,j])
              while(abs(q1[3,j]) >= q_3.upper) {
                if(q1[3,j] <= -q_3.upper) {
                  q1[3,j] <- -2*q_3.upper - q1[3,j]; p1[3,j] <- -p1[3,j]
                } else {
                  q1[3,j] <- 2*q_3.upper - q1[3,j]; p1[3,j] <- -p1[3,j]
                }
              }

              while(q1[4,j] < 0 || q1[4,j] >= 2*pi) {
                if(q1[4,j] < 0) {
                  q1[4,j] <- - q1[4,j]; p1[4,j] <- -p1[4,j]
                } else {
                  q1[4,j] <- 4*pi - q1[4,j]; p1[4,j] <- -p1[4,j]
                }
              }
              while(q1[5,j] < 0 || q1[5,j] >= 2*pi) {
                if(q1[5,j] < 0) {
                  q1[5,j] <- - q1[5,j]; p1[5,j] <- -p1[5,j]
                } else {
                  q1[5,j] <- 4*pi - q1[5,j]; p1[5,j] <- -p1[5,j]
                }
              }
            }


            p <- p1; q <- q1
          }
          # Make a full step for the momentum, except at end of trajectory

          if(any(is.nan(c(p, q)))) {
            broken <- TRUE
            break
          } else if (i!=L)  {
            p <- p - epsilon * (- grad_wnorm2_all_comp(data.rad, q, pi.mix.1, omega.2pi, ncores)
                                + matrix(c(1/gam.scale + (1- 1/gam.scale)/q[1:2,], q[3,], rep(0,2)), ncol=1 )) # the second term in the bracket arises from prior
          }
        }

        if(all(!broken, !kappa.large)) {
          if(any(is.nan(c(p,q))))  {
            broken <- TRUE
          } else {
            # Make a half step for momentum at the end.

            p <- p - (epsilon/2) * (- grad_wnorm2_all_comp(data.rad, q, pi.mix.1, omega.2pi, ncores)
                                    + matrix(c(1/gam.scale + (1- 1/gam.scale)/q[1:2,], q[3,], rep(0,2)), ncol=1 )) # the second term in the bracket arises from prior
          }
        }

        # Negate momentum at end of trajectory to make the proposal symmetric

        if(any(is.nan(c(p, q))))  {
          broken <- TRUE
        } else {
          p <-  -p
        }
      }

      if (iter > 100 && mean(accpt.par.mat.all[1:iter]) < 0.05) {
        broken <- TRUE
      }

      if (broken) {
        print("Acceptance rate too low. Automatically restarting with a smaller \'epsilon\'.")
        iter <- 2
        if(epsilon.random) {
          epsilon_vec <- epsilon_vec/2
        } else {
          epsilon <- epsilon/2
        }


        par.mat.all[,,iter] <- par.mat.1
        pi.mix.all[,iter] <- pi.mix.1
        llik.all[iter] <- llik.1
        lprior.all[iter] <- lprior.1
        lpd.all[iter] <- lpd.1
        accpt.par.mat.all[iter] <- accpt.par.mat


        next

      }

      # Evaluate potential and kinetic energies at start and end of trajectory

      current_U <- -lpd.1
      current_K <- sum(current_p^2) / 2

      if(kappa.large) {
        proposed_U <- proposed_K <- Inf
      } else {
        par.mat.prop <- q
        l.c.wnorm2.prop <- log_const_wnorm2_all(par.mat.prop)

        lprior.prop <- sum(ldgamanum(q[1:2,], gam.loc, gam.scale)) - 0.5*sum((q[3,]/norm.var)^2)

        llik.prop <- llik_wnorm2_full(data.rad, q, pi.mix.1, l.c.wnorm2.prop, omega.2pi, ncores)

        proposed_U <- -(llik.prop + lprior.prop)
        proposed_K <- sum(p^2) / 2
      }

      exp(current_U-proposed_U+current_K-proposed_K)
      # Accept or reject the state at end of trajectory, returning either
      # the position at the end of the trajectory or the initial position

      if (runif(1) < exp(current_U-proposed_U+current_K-proposed_K))
      {
        # return (q)  # accept
        # accpt = 1
        par.mat.1 <- signif(par.mat.prop, 8)
        lprior.1 <- signif(lprior.prop, 8)
        llik.1 <- signif(llik.prop, 8)
        lpd.1 <- signif(-proposed_U, 8)
        accpt.par.mat <- 1
        l.c.wnorm2.1 <- signif(l.c.wnorm2.prop, 8)
      }


      MC <- list("par.mat" = par.mat.1, "pi.mix" = pi.mix.1,
                 "l.c.wnorm2" = l.c.wnorm2.1, "llik" = llik.1, "lprior" = lprior.1, "lpd" = lpd.1,
                 "accpt.par.mat" = accpt.par.mat)

      par.mat.all[,,iter] <- MC$par.mat
      pi.mix.all[,iter] <- MC$pi.mix
      llik.all[iter] <- llik.1
      lprior.all[iter] <- lprior.1
      lpd.all[iter] <- lpd.1
      accpt.par.mat.all[iter] <- accpt.par.mat


      # tuning epsilon with first 20 draws
      if(autotune && iter == iter.tune && mean(accpt.par.mat.all[2:(iter.tune+1)]) < 0.6) {
        iter <- 2
        ntune <- ntune + 1
        if(epsilon.random) {
          epsilon_vec <- epsilon_vec/2
        } else {
          epsilon <- epsilon/2
        }
      }

      if(show.progress && ((iter-1) %% 25 == 0 || iter == n.iter + 1))
        utils::setTxtProgressBar(pb, iter)

      iter <- iter + 1

    }
  }

  if(ncomp == 1 && grepl(method, "rwmh")) # using rwmh
  {
    while(iter <= (n.iter+1)) {
      pi.mix.1 <- 1
      par.mat.old <- MC$par.mat
      l.c.wnorm2.old <- MC$l.c.wnorm2
      llik_new.pi <- MC$llik

      #----------------------------------------------------------------------------------
      #generating presicion parameters
      #----------------------------------------------------------------------------------

      k1.1.old <- par.mat.old[1, ]
      k2.1.old <- par.mat.old[2, ]
      k3.1.old <- par.mat.old[3, ]

      k1.1.prop <- pmax(k1.1.old + rnorm(ncomp,0,propscale[1]), 1e-6)
      k2.1.prop <- pmax(k2.1.old + rnorm(ncomp,0,propscale[2]), 1e-6)
      k3.1.prop <- k3.1.old + rnorm(ncomp,0,propscale[3])

      prop.mat <- unname(matrix(c(k1.1.prop,k2.1.prop,k3.1.prop, par.mat.old[4:5, ]),ncol=1))

      llik_old <- llik_new.pi
      lprior_old <- MC$lprior

      if(any((k1.1.prop * k2.1.prop) - (k3.1.prop * k3.1.prop) <= 0)) {
        llik_prop <- lprior_prop <- -Inf
      } else {
        l.c.wnorm2.prop <- as.numeric(log_const_wnorm2_all(prop.mat))
        llik_prop <- llik_wnorm2_full(data.rad, prop.mat, pi.mix.1, l.c.wnorm2.prop, omega.2pi, ncores)
        lprior_prop <- sum(ldgamanum(prop.mat[1:2,], gam.loc, gam.scale)) - 0.5*sum((prop.mat[3,]/norm.var)^2)
      }

      post.omg_old <- llik_old + lprior_old
      post.omg_prop <- llik_prop + lprior_prop

      if (runif(1) <  exp(post.omg_prop-post.omg_old) ) {
        k1.1 <- k1.1.prop
        k2.1 <- k2.1.prop
        k3.1 <- k3.1.prop
        accpt.kappa <- 1
        l.c.wnorm2.1 <- signif(l.c.wnorm2.prop, 8)
        llik_new.omg <- signif(llik_prop, 8)
        lprior.1 <- signif(lprior_prop, 8)
        par.mat_new.omg <- signif(prop.mat, 8)
      } else {
        k1.1 <- k1.1.old
        k2.1 <- k2.1.old
        k3.1 <- k3.1.old
        accpt.kappa <- 0
        l.c.wnorm2.1 <- l.c.wnorm2.old
        llik_new.omg <- llik_old
        lprior.1 <- lprior_old
        par.mat_new.omg <- par.mat.old
      }


      #----------------------------------------------------------------------------------
      #generating mu and nu
      #----------------------------------------------------------------------------------
      prop.mu <- prncp_reg(par.mat.old[4, ] + rnorm(ncomp,0,propscale[4]))
      prop.nu <- prncp_reg(par.mat.old[5, ] + rnorm(ncomp,0,propscale[5]))
      #----------------------------------------------------------------------------------
      prop.mat.mean <- matrix(c(par.mat_new.omg[1:3,], prop.mu,prop.nu), ncol=1)

      llik_new.prop <- llik_wnorm2_full(data.rad, prop.mat.mean, pi.mix.1, l.c.wnorm2.1, omega.2pi, ncores)

      if (runif(1) <  exp(llik_new.prop-llik_new.omg) ) {
        par.mat.1 <- signif(prop.mat.mean, 8)
        accpt.mu <- 1
        llik.1 <- signif(llik_new.prop, 8)
      } else {
        par.mat.1 <- par.mat_new.omg
        accpt.mu <- 0
        llik.1 <- llik_new.omg
      }

      lpd.1 <- llik.1 + lprior.1

      MC <- list("par.mat" = par.mat.1, "pi.mix" = pi.mix.1,
                 "l.c.wnorm2" = l.c.wnorm2.1, "llik" = llik.1, "lprior" = lprior.1, "lpd" = lpd.1,
                 "accpt.kappa" = accpt.kappa, "accpt.mu" = accpt.mu)

      par.mat.all[,,iter] <- MC$par.mat
      pi.mix.all[,iter] <- MC$pi.mix
      llik.all[iter] <- llik.1
      lprior.all[iter] <- lprior.1
      lpd.all[iter] <- lpd.1
      accpt.kappa.all[iter] <- accpt.kappa
      accpt.mu.all[iter] <- accpt.mu

      # tuning propscale with first 20 draws
      if(autotune && iter == iter.tune && (mean(accpt.kappa.all[2:(iter.tune+1)]) < 0.6 ||
                                           mean(accpt.mu.all[2:(iter.tune+1)]) < 0.6)) {
        iter <- 2
        ntune <- ntune + 1
        propscale <- propscale/2
      }

      if(show.progress && ((iter-1) %% 25 == 0 || iter == n.iter + 1))
        utils::setTxtProgressBar(pb, iter)

      iter <- iter+1

    }
  }
  #******************************************************************************************

  #******************************************************************************************
  # multiple component model
  #******************************************************************************************

  if(ncomp > 1 && grepl(method, "hmc")) # using hmc
  {
    if(epsilon.random)
      epsilon_vec <- runif(n.iter, min = 0.9*epsilon, max = 1.1*epsilon)
    if(L.random)
      L_vec <- sample(1:L, n.iter, replace = TRUE)


    while(iter <= (n.iter+1)) {
      broken <- FALSE
      kappa.large <- FALSE

      #----------------------------------------------------------------------------------
      #generating mixture proportions
      #----------------------------------------------------------------------------------
      pi.mix.old <- MC$pi.mix
      par.mat.old <- MC$par.mat
      l.c.wnorm2.old <- MC$l.c.wnorm2


      # Gibbs Sampler
      {
        post.wt <- mem_p_wnorm2(data.rad, par.mat.old, pi.mix.old, l.c.wnorm2.old, omega.2pi, ncores)
        clus.ind[ , iter] <- cID(post.wt, ncomp, runif(n.data))
        n.clus <- tabulate(clus.ind[ , iter], nbins = ncomp) #vector of component sizes
        pi.mix.1 <- as.numeric(rdirichlet(1, (1 + n.clus))) #new mixture proportions
        llik_new.pi <- llik_wnorm2_full(data.rad, par.mat.old, pi.mix.1, l.c.wnorm2.old, omega.2pi, ncores)
      }

      #----------------------------------------------------------------------------------
      #generating par.mat by HMC
      #----------------------------------------------------------------------------------

      par.mat.1 <- par.mat.old
      lprior.1 <- MC$lprior
      llik.1 <- llik_new.pi
      lpd.1 <- llik.1 + lprior.1
      l.c.wnorm2.1 <- MC$l.c.wnorm2
      accpt.par.mat <- 0


      current_q <- par.mat.1
      current_p <- matrix(rnorm(5*ncomp,0,1), nrow = 5)  # independent standard normal variates
      p <- current_p
      q <- current_q

      if(L.random)
        L <- L_vec[iter-1]

      if(epsilon.random)
        epsilon <- epsilon_vec[iter-1]

      # Do leapfrog with L and epsilon
      {
        # Make a half step for momentum at the beginning

        p <- p - (epsilon/2) * (- grad_wnorm2_all_comp(data.rad, q, pi.mix.1, omega.2pi, ncores)
                                + rbind((1/gam.scale + (1- 1/gam.scale)/q[1:2,]), q[3,], matrix(0, nrow = 2, ncol = ncomp)) ) # the second term in the bracket arises from prior
        # Alternate full steps for position and momentum

        for (i in 1:L)
        {
          # Make a full step for the position

          q <- q + epsilon * p

          if(all(!is.nan(q)) && any(abs(q[1:3, ]) >= kappa_upper)) {
            kappa.large <- TRUE
            break
          }

          if(any(is.nan(c(q,p)))) {
            broken <- TRUE
            #stop("Algorithm breaks. Try a smaller epsilon.")
            break
          }
          # Make sure the components of q1 are in the proper ranges
          {
            q1 <- q; p1 <- p

            for(j in 1:ncomp) {

              while(q1[1,j] <= 0) {
                q1[1,j] <- -q1[1,j]; p1[1,j] <- -p1[1,j]
              }

              while(q1[2,j] <= 0) {
                q1[2,j] <- -q1[2,j]; p1[2,j] <- -p1[2,j]
              }

              q_3.upper <- sqrt(q1[1,j]*q1[2,j])
              while(abs(q1[3,j]) >= q_3.upper) {
                if(q1[3,j] <= -q_3.upper) {
                  q1[3,j] <- -2*q_3.upper - q1[3,j]; p1[3,j] <- -p1[3,j]
                } else {
                  q1[3,j] <- 2*q_3.upper - q1[3,j]; p1[3,j] <- -p1[3,j]
                }
              }

              while(q1[4,j] < 0 || q1[4,j] >= 2*pi) {
                if(q1[4,j] < 0) {
                  q1[4,j] <- - q1[4,j]; p1[4,j] <- -p1[4,j]
                } else {
                  q1[4,j] <- 4*pi - q1[4,j]; p1[4,j] <- -p1[4,j]
                }
              }
              while(q1[5,j] < 0 || q1[5,j] >= 2*pi) {
                if(q1[5,j] < 0) {
                  q1[5,j] <- - q1[5,j]; p1[5,j] <- -p1[5,j]
                } else {
                  q1[5,j] <- 4*pi - q1[5,j]; p1[5,j] <- -p1[5,j]
                }
              }
            }


            p <- p1; q <- q1
          }
          # Make a full step for the momentum, except at end of trajectory

          if(any(is.nan(c(q, p)))) {
            broken <- TRUE
            #stop("Algorithm breaks. Try a smaller epsilon.")
            break
          } else if (i!=L) {
            p <- p - epsilon * (- grad_wnorm2_all_comp(data.rad, q, pi.mix.1, omega.2pi, ncores)
                                + rbind((1/gam.scale + (1- 1/gam.scale)/q[1:2,]), q[3,], matrix(0, nrow = 2, ncol = ncomp)) ) # the second term in the bracket arises from prior

          }
        }

        # Make a half step for momentum at the end.
        if(all(!broken, !kappa.large)){
          if(any(is.nan(c(p, q)))) {
            broken <- TRUE
          } else {
            p <- p - (epsilon/2) * (- grad_wnorm2_all_comp(data.rad, q, pi.mix.1, omega.2pi, ncores)
                                    + rbind((1/gam.scale + (1- 1/gam.scale)/q[1:2,]), q[3,], matrix(0, nrow = 2, ncol = ncomp)) ) # the second term in the bracket arises from prior
          }
        }

        if(any(is.nan(c(p, q)))) {
          broken <- TRUE
        } else {
          # Negate momentum at end of trajectory to make the proposal symmetric

          p <-  -p
        }
      }

      if (iter > 100 && mean(accpt.par.mat.all[1:iter]) < 0.05) {
        broken <- TRUE
      }

      if (broken) {
        print("Acceptance rate too low. Automatically restarting with a smaller \'epsilon\'.")
        iter <- 2
        if(epsilon.random) {
          epsilon_vec <- epsilon_vec/2
        } else {
          epsilon <- epsilon/2
        }


        par.mat.all[,,iter] <- par.mat.1
        pi.mix.all[,iter] <- pi.mix.old
        llik.all[iter] <- llik.1
        lprior.all[iter] <- lprior.1
        lpd.all[iter] <- lpd.1
        accpt.par.mat.all[iter] <- accpt.par.mat


        next

      }


      # Evaluate potential and kinetic energies at start and end of trajectory

      current_U <- -lpd.1
      current_K <- sum(current_p^2) / 2

      if(kappa.large) {
        proposed_U <- proposed_K <- Inf
      } else {
        par.mat.prop <- q
        l.c.wnorm2.prop <- log_const_wnorm2_all(par.mat.prop)

        lprior.prop <- sum(ldgamanum(q[1:2,], gam.loc, gam.scale)) - 0.5*sum((q[3,]/norm.var)^2)

        llik.prop <- llik_wnorm2_full(data.rad, q, pi.mix.1, l.c.wnorm2.prop, omega.2pi, ncores)

        proposed_U <- -(llik.prop + lprior.prop)
        proposed_K <- sum(p^2) / 2

      }
      exp(current_U-proposed_U+current_K-proposed_K)
      # Accept or reject the state at end of trajectory, returning either
      # the position at the end of the trajectory or the initial position

      if (runif(1) < exp(current_U-proposed_U+current_K-proposed_K))
      {
        # return (q)  # accept
        # accpt = 1
        par.mat.1 <- signif(par.mat.prop, 8)
        lprior.1 <- signif(lprior.prop, 8)
        llik.1 <- signif(llik.prop, 8)
        lpd.1 <- signif(-proposed_U, 8)
        accpt.par.mat <- 1
        l.c.wnorm2.1 <- signif(l.c.wnorm2.prop, 8)
      }


      MC <- list("par.mat" = par.mat.1, "pi.mix" = pi.mix.1,
                 "l.c.wnorm2" = l.c.wnorm2.1, "llik" = llik.1, "lprior" = lprior.1, "lpd" = lpd.1,
                 "accpt.par.mat" = accpt.par.mat)

      par.mat.all[,,iter] <- MC$par.mat
      pi.mix.all[,iter] <- MC$pi.mix
      llik.all[iter] <- llik.1
      lprior.all[iter] <- lprior.1
      lpd.all[iter] <- lpd.1
      accpt.par.mat.all[iter] <- accpt.par.mat


      # tuning epsilon with first 20 draws
      if(autotune && iter == iter.tune && mean(accpt.par.mat.all[2:(iter.tune+1)]) < 0.6) {
        iter <- 2
        ntune <- ntune + 1
        if(epsilon.random) {
          epsilon_vec <- epsilon_vec/2
        } else {
          epsilon <- epsilon/2
        }
      }

      if(show.progress && ((iter-1) %% 25 == 0 || iter == n.iter + 1))
        utils::setTxtProgressBar(pb, iter)

      iter <- iter + 1

    }
  }

  if(ncomp > 1 && grepl(method, "rwmh")) # using rwmh
  {
    while(iter <= (n.iter+1)) {
      #----------------------------------------------------------------------------------
      #generating mixture proportions
      #----------------------------------------------------------------------------------
      pi.mix.old <- MC$pi.mix
      par.mat.old <- MC$par.mat
      l.c.wnorm2.old <- MC$l.c.wnorm2


      # Gibbs Sampler
      {
        post.wt <- mem_p_wnorm2(data.rad, par.mat.old, pi.mix.old, l.c.wnorm2.old, omega.2pi, ncores)
        clus.ind[ , iter] <- cID(post.wt, ncomp, runif(n.data))
        n.clus <- tabulate(clus.ind[ , iter], nbins = ncomp) #vector of component sizes
        pi.mix.1 <- as.numeric(rdirichlet(1, (1 + n.clus))) #new mixture proportions
        llik_new.pi <- llik_wnorm2_full(data.rad, par.mat.old, pi.mix.1, l.c.wnorm2.old, omega.2pi, ncores)
      }


      #----------------------------------------------------------------------------------
      #generating presicion parameters
      #----------------------------------------------------------------------------------

      k1.1.old <- par.mat.old[1, ]
      k2.1.old <- par.mat.old[2, ]
      k3.1.old <- par.mat.old[3, ]

      k1.1.prop <- pmax(k1.1.old + rnorm(ncomp,0,propscale[1]), 1e-6)
      k2.1.prop <- pmax(k2.1.old + rnorm(ncomp,0,propscale[2]), 1e-6)
      k3.1.prop <- k3.1.old + rnorm(ncomp,0,propscale[3])

      prop.mat <- unname(rbind(k1.1.prop,k2.1.prop,k3.1.prop, par.mat.old[4:5, ]))

      llik_old <- llik_new.pi
      lprior_old <- MC$lprior

      if(any((k1.1.prop * k2.1.prop) - (k3.1.prop * k3.1.prop) <= 0)) {
        llik_prop <- lprior_prop <- -Inf
      } else {
        l.c.wnorm2.prop <- as.numeric(log_const_wnorm2_all(prop.mat))
        llik_prop <- llik_wnorm2_full(data.rad, prop.mat, pi.mix.1, l.c.wnorm2.prop, omega.2pi, ncores)
        lprior_prop <- sum(ldgamanum(prop.mat[1:2,], gam.loc, gam.scale)) - 0.5*sum((prop.mat[3,]/norm.var)^2)
      }


      post.omg_old <- llik_old + lprior_old
      post.omg_prop <- llik_prop + lprior_prop

      if (runif(1) <  exp(post.omg_prop-post.omg_old) ) {
        k1.1 <- k1.1.prop
        k2.1 <- k2.1.prop
        k3.1 <- k3.1.prop
        accpt.kappa <- 1
        l.c.wnorm2.1 <- signif(l.c.wnorm2.prop, 8)
        llik_new.omg <- signif(llik_prop, 8)
        lprior.1 <- signif(lprior_prop, 8)
        par.mat_new.omg <- signif(prop.mat, 8)
      } else {
        k1.1 <- k1.1.old
        k2.1 <- k2.1.old
        k3.1 <- k3.1.old
        accpt.kappa <- 0
        l.c.wnorm2.1 <- l.c.wnorm2.old
        llik_new.omg <- llik_old
        lprior.1 <- lprior_old
        par.mat_new.omg <- par.mat.old
      }


      #----------------------------------------------------------------------------------
      #generating mu and nu
      #----------------------------------------------------------------------------------
      prop.mu <- prncp_reg(par.mat.old[4, ] + rnorm(ncomp,0,propscale[4]))
      prop.nu <- prncp_reg(par.mat.old[5, ] + rnorm(ncomp,0,propscale[5]))
      #----------------------------------------------------------------------------------
      prop.mat.mean <- unname(rbind(par.mat_new.omg[1:3,], prop.mu,prop.nu))

      llik_new.prop <- llik_wnorm2_full(data.rad, prop.mat.mean, pi.mix.1, l.c.wnorm2.1, omega.2pi, ncores)

      if (runif(1) <  exp(llik_new.prop-llik_new.omg) ) {
        par.mat.1 <- signif(prop.mat.mean, 8)
        accpt.mu <- 1
        llik.1 <- signif(llik_new.prop, 8)
      } else {
        par.mat.1 <- par.mat_new.omg
        accpt.mu <- 0
        llik.1 <- llik_new.omg
      }

      lpd.1 <- llik.1 + lprior.1

      MC <- list("par.mat" = par.mat.1, "pi.mix" = pi.mix.1,
                 "l.c.wnorm2" = l.c.wnorm2.1, "llik" = llik.1, "lprior" = lprior.1,
                 "lpd" = lpd.1, "accpt.kappa" = accpt.kappa, "accpt.mu" = accpt.mu)

      par.mat.all[,,iter] <- MC$par.mat
      pi.mix.all[,iter] <- MC$pi.mix
      llik.all[iter] <- llik.1
      lprior.all[iter] <- lprior.1
      lpd.all[iter] <- lpd.1
      accpt.kappa.all[iter] <- accpt.kappa
      accpt.mu.all[iter] <- accpt.mu

      # tuning propscale with first 20 draws
      if(autotune && iter == iter.tune && (mean(accpt.kappa.all[2:(iter.tune+1)]) < 0.6 ||
                                           mean(accpt.mu.all[2:(iter.tune+1)]) < 0.6)) {
        iter <- 2
        ntune <- ntune + 1
        propscale <- propscale/2
      }

      if(show.progress && ((iter-1) %% 25 == 0 || iter == n.iter + 1))
        utils::setTxtProgressBar(pb, iter)

      iter <- iter+1

    }
  }
  #******************************************************************************************
  if(grepl(method, "hmc")) {
    if(epsilon.random) {
      epsilon_ave <- mean(epsilon_vec)
    } else{
      epsilon_ave <- epsilon
    }
    if(L.random) {
      L_ave <- mean(L_vec)
    } else{
      L_ave <- L
    }
  }

  if(grepl(method, "rwmh")) {
    propscale_final <- propscale
  }

  if(show.progress)  cat("\n")

  allpar_val <- array(1, dim = c(6, ncomp, n.iter+1))
  allpar_val[1, , ] <- pi.mix.all
  allpar_val[2:6, , ] <- par.mat.all

  allpar_name <- c("pmix", modelpar.names)
  dimnames(allpar_val)[[1]] <- c("pmix", modelpar.names)

  result <- list("par.value" = allpar_val, "par.name" = allpar_name, "llik" = llik.all,
                 "accpt.modelpar" = accpt.par.mat.all,
                 "accpt.kappa" = accpt.kappa.all, "accpt.mu" = accpt.mu.all,
                 "lpd" = lpd.all, "model" = curr.model, "method" = method, "clus.ind" = clus.ind,
                 "epsilon.random" = epsilon.random, "epsilon" = epsilon_ave,
                 "L.random" = L.random, "L" = L_ave, "type" = "bi", "omega.2pi" = omega.2pi,
                 "propscale.final" = propscale_final, "data" = data.rad, "int.displ" = int.displ,
                 "n.data" = n.data, "ncomp" = ncomp, "n.iter" = n.iter)
  class(result) <- "angmcmc"

  return(result)
}
