#' Parallel Vorob'ev criterion
#'
#' Evaluation of the parallel Vorob'ev criterion for some candidate points. To be used in optimization routines, like in \code{\link{max_vorob_parallel_gpc}}. To avoid numerical instabilities, the new points are evaluated only if they are not too close to an existing observation, or if there is some observation noise. The criterion is the integral of the posterior Vorob'ev uncertainty.
#'
#' @param x input vector of size \code{batchsize*d} at which one wants to evaluate the criterion. This argument is NOT a matrix.
#' @param integration.points matrix of points for numerical integration in the design space.
#' @param integration.weights vector of size \code{ncol(integration.points)} corresponding to the weights of these integration points.
#' @param intpoints.oldmean (see below).
#' @param intpoints.oldsd vectors of size \code{ncol(integration.points)} corresponding to the mean and standard deviation at the integration points before adding the batchsize points \code{x} to the design of experiments.
#' @param precalc.data list containing precalculated data. This list can be generated using the \code{precomputeUpdateData} function. 
#' @param object object of class \code{\link[GPCsign]{gpcm}}.
#' @param new.noise.var optional scalar value of the noise variance for the new observations.
#' @param batchsize number of points to sample simultaneously. The sampling criterion will return batchsize points at a time for sampling.
#' @param alpha a scalar representing the Vorob'ev threshold
#' @param current.vorob current value of the vorob criterion (before adding new observations).
#' @param seed to fix the seed.
#'
#' @return Parallel vorob value
#'
#' @export
#'
#' @import GPCsign
#'
#' @references Menz, M., Munoz-Zuniga, M., Sinoquet, D. Estimation of simulation failure set with active learning based on Gaussian Process classifiers and random set theory (2023). \url{https://hal.science/hal-03848238}.
#' @references Chevalier, C. Fast uncertainty reduction strategies relying on Gaussian process models PhD Thesis. University of Bern (2013).
#' @references El Amri, M.R. Analyse d’incertitudes et de robustesse pour les modèles à entrées et sorties fonctionnelles. Theses, Université Grenoble Alpes (2019), \url{https://theses.hal.science/tel-02433324}.
#' @references El Amri, M.R., Helbert, C., Munoz-Zuniga, M. Feasible set estimation under functional uncertainty by gaussian process modelling (2023).  455:133,893. \doi{10.1016/j.physd.2023.133893}.
#' @references Bachoc, F., Helbert, C. & Picheny, V. Gaussian process optimization with failures: classification and convergence proof. \emph{J Glob Optim} \bold{78}, 483–506 (2020). \doi{10.1007/s10898-020-00920-0}.
#'
#' @author Morgane MENZ, Delphine SINOQUET, Miguel MUNOZ-ZUNIGA. Contributors: Naoual SERRAJI.
#'
#' @examples

#' #-------------------------------------------------------------------
#' #-------------------vorob_optim_parallel_gpc------------------------
#' #-------------------------------------------------------------------
#'
#' ## 20-points DoE, and the corresponding response
#' d <- 2
#' nb_PX <- 20
#' x_ <- matrix(c(0.205293785978832, 0.0159983370750337,
#'               0.684774733109666, 0.125251417595962,
#'               0.787208786290006, 0.700475706055049,
#'               0.480507717105934, 0.359730889653793,
#'               0.543665267336735, 0.565974761807069,
#'               0.303412043992361, 0.471502352650857,
#'               0.839505250127309, 0.504914690245002,
#'               0.573294917143728, 0.784444726564573,
#'               0.291681289223421, 0.255053812451938,
#'               0.87233450888786, 0.947168337730927,
#'               0.648262257638515, 0.973264712407035,
#'               0.421877310273815, 0.0686662506387988,
#'               0.190976166753807, 0.810964668176754,
#'               0.918527262507395, 0.161973686467513,
#'               0.0188128700859558, 0.43522031347403,
#'               0.99902788789426, 0.655561821513544,
#'               0.741113863862512, 0.321050086076934,
#'               0.112003007565305, 0.616551317575545,
#'               0.383511473487687, 0.886611679106771,
#'               0.0749211435982952, 0.205805968972305),
#'             byrow = TRUE, ncol = d)
#' require(DiceKriging)
#' fx <- apply(x_, 1, branin)
#' f <- ifelse(fx < 14, -1, 1)
#' Xf <- as.matrix(x_)
#'
#' require(future)
#' plan(multisession)
#' ## gpcm object
#' require(GPCsign)
#' model <- gpcm(f, Xf, coef.m = -1.25, coef.cov = c(1.17,0.89))
#'
#' ## parameteres for vorob_optim_parallel_gpc function
#' batchsize <- 1
#' x <- matrix(c(0.1,0.2),ncol=2,byrow=TRUE)
#' require(randtoolbox)
#' nb.integration <- d * 1000
#' integration.points <- sobol(n = nb.integration, dim = d, scrambling = 0)
#' integration.points <- scale(x = integration.points, center = model@X.mean, scale = model@X.std)
#' precalc.data <- precomputeUpdateData(model=model, integration.points=integration.points)
#' intpoints.oldmean <- precalc.data$intpoints.oldmean
#' intpoints.oldsd <- precalc.data$intpoints.oldsd
#' 
#' pn <- precalc.data$pn
#' 
#' require(KrigInv)
#' alpha <- KrigInv::vorob_threshold(pn)
#' pn_bigger_than_alpha <- (pn>alpha)+0
#' pn_lower_than_alpha <- 1-pn_bigger_than_alpha
#' penalisation <- 1
#' current.vorob <- mean(pn*pn_lower_than_alpha + penalisation*(1-pn)*pn_bigger_than_alpha)
#'
#' x <- scale(x = x, center = model@X.mean, scale = model@X.std)
#' criter <- vorob_optim_parallel_gpc(x = x, integration.points = integration.points,
#'                                    intpoints.oldmean = intpoints.oldmean,
#'                                    intpoints.oldsd = intpoints.oldsd,
#'                                    precalc.data = precalc.data,
#'                                    object = model,
#'                                    batchsize = batchsize,
#'                                    alpha = alpha,
#'                                    current.vorob = current.vorob)
#' plan(sequential)
#' 
vorob_optim_parallel_gpc <- function(x, integration.points,integration.weights=NULL,
                                 intpoints.oldmean,intpoints.oldsd,precalc.data,
                                 object, new.noise.var=NULL,batchsize,alpha,current.vorob,
                                 seed=NULL){


  if(!is.null(new.noise.var)){
    if(new.noise.var == 0) {
      new.noise.var <- NULL
    }
  }
  # x is a vector of size d * batchsize
  Xf <- object@X
  d <- object@d
  X.new <- matrix(x,ncol=d)
  n <- nrow(object@X)
  N_int <- nrow(integration.points)
  q <-  nrow(X.new) # q <- batchsize


  mindist <- Inf
  maxdist <- 0.


  if(!is.null(object@X.mean)){
    X.new.normalize <- scale(x = X.new, center = object@X.mean, scale = object@X.std)
    x <- as.numeric(t(X.new.normalize))
    Xf <- scale(x = object@X, center = object@X.mean, scale = object@X.std)
    }
  else{
    X.new.normalize<-X.new}

  tp1 <- c(as.numeric(t(Xf)),as.numeric(x))
  for (i in 1:batchsize){
    #distance between the i^th point and all other points (in the DoE or in the batch) in normalized space (if normalize=T)
    xx <- X.new.normalize[i,]
    tp2<-matrix(tp1-as.numeric(xx),ncol=d,byrow=TRUE)^2
    mysums <- sqrt(rowSums(tp2))
    maxdist <- max(mysums)
    mysums[n+i] <- Inf #because this one is usually equal to zero...
    mindist <- min(mindist,mysums)
  }

  gpc <- predict(object=object, newdata=X.new)
  Zmean.newdata <-  gpc$Zsimu_mean
  Zvar.newdata <-  gpc$Zsimu_var
  cov.newdata <-  gpc$cov # conditional covariance of X.new
  c.newdata <-  gpc$c # unconditional covariance between X.new and design points

  if (!identical(colnames(integration.points), colnames(object@X))) colnames(integration.points) <- colnames(object@X)
  if ( mindist > 1e-4 || (!is.null(new.noise.var))){

    if(!is.null(object@X.mean)){
            X.new <- X.new.normalize
            integration.points <- scale(x = integration.points, center = object@X.mean, scale = object@X.std)
            }
    kn <- computeQuickgpccov(object,integration.points,X.new,precalc.data, c.newdata)

    # compute sd new and t(lambda_new)
    krig2 <- predict_update_gpc_parallel(Sigma.r = cov.newdata, newdata.oldsd = intpoints.oldsd, kn = kn)

    if(!is.null(krig2$error)) return(current.vorob)
    sk.new <- krig2$sd

    term1 <- 0
    term3 <- 0
    term2 <-0
    ## v2

    vres <- sapply(X= seq(1:(2^q)),FUN=computeVorobTerms,object=object,intpoints.oldmean=intpoints.oldmean,krig2=krig2,sk.new=sk.new,alpha=alpha,gpc=gpc,X.new=X.new, seed=seed)

    for(u in vres[1,]) term1 <- term1 + u
    for(u in vres[2,]) term3 <- term3 + u
    for(u in vres[3,]) term2 <- term2 + u
    result = 2*term1 - term2 + term3


    if (is.null(integration.weights)) {crit <- mean(result)
    }else crit <- sum(result*integration.weights)
  }else crit <- current.vorob + 0.01
  return(crit)
}


