#' Run ARCHISSUR algorithm
#'
#' \code{archissur} adaptively enriches the Gaussian Process Classifier (GPC) using a learning criterion to achieve a precise approximation of the feasible area contour. This is done by iteratively adding the best learning point that minimizes future uncertainty over the feasible domain, following the Stepwise Uncertainty Reduction strategy (SUR).
#'
#' @param design.init optional matrix representing the initial design of experiments (DoE). If not provided, you must provide a \code{model} of type \code{gpcm}.
#' @param cst.init optional vector of binary observations \{0,1\} corresponding to the initial class labels. If not provided, it will be calculated using \code{cst_function}.
#' @param model optional object of type \code{gpcm} to start archissur. If not provided, you must provide an initial DoE \code{design.init}.
#' @param cst_function constraint function with binary outputs \{0,1\} to be learn.
#' @param lower inputs lower bound of \code{design.init}.
#' @param upper inputs upper bound of \code{design.init}.
#' @param n.ite number of iterations of \code{archissur}.
#' @param seed to fix the seed.
#' @param nb.integration number of integration points. Default is \code{d*1000}.
#' @param plot_2D_pn if \code{TRUE} and \code{d = 2}, plot class 1 probability map and learning points. Plots are available in '2D_plots' directory. Default is \code{FALSE}.
#' @param batchsize number of points to be learned at each iteration. Default is 1.
#' @param n_update number of iterations between hyperparameter updates by likelihood maximization. Default is 1.
#' @param gpc.options list with GPC model options:  covariance kernel, noise variance, number of initial points for MLE optimization, standardization of inputs, constrained latent GP mean sign.If \code{NULL}, default options are \code{list(normalize = T, multistart = 1, covtype = "matern5_2", MeanTransform=NULL)}. Default is \code{NULL}.
#' @param optimcontrol optional list of control parameters for enrichment criterion optimization. Default is \code{NULL}. The field "\code{method}" defines which optimization method is used: it can be either \code{"multistart"} (default) for an optimization with \code{L-BFGS-B} multistart, or \code{"discrete"} for an optimization over a specified discrete set, or \code{"genoud"} for an optimization using the genoud algorithm. (See details).
#' @param verbose Level of verbosity for printing information during iterations. 0: No printing. 1: Print iteration number and best point found. 2: Print iteration number, best point found, criterion value, and model hyperparameters. Default is 1.
#'        
#' @return
#' A list containing:
#' \item{Xf}{DoE}
#' \item{f}{binary observations corresponding to the class labels.}
#' \item{alpha}{a scalar representing the Vorob'ev threshold.}
#' \item{vorob_expect}{Vorob’ev expectation.}
#' \item{vorob_dev}{current Vorob’ev deviation.}
#' \item{model}{an object of class \code{\link[GPCsign]{gpcm}} containing the GPC model at last iteration.}
#' An '.Rds' file \code{model_gp.Rds} containing an object of class \code{\link[GPCsign]{gpcm}} corresponding to current GPC model.
#' The class 1 probability map. Plots are available in '2D_plots' directory.
#'
#'
#' @importFrom GPCsign gpcm 
#' @importFrom randtoolbox sobol
#' @importFrom grDevices dev.off hcl.colors png
#' @importFrom graphics axis filled.contour points
#' @importMethodsFrom GPCsign predict
#' @importMethodsFrom GPCsign update
#' @importFrom KrigInv vorob_threshold
#'
#' @import GPCsign
#'
#' @export
#'
#' @references Menz, M., Munoz-Zuniga, M., Sinoquet, D. Estimation of simulation failure set with active learning based on Gaussian Process classifiers and random set theory (2023). \url{https://hal.science/hal-03848238}.
#' @references Bachoc, F., Helbert, C. & Picheny, V. Gaussian process optimization with failures: classification and convergence proof. \emph{J Glob Optim} \bold{78}, 483–506 (2020). \doi{10.1007/s10898-020-00920-0}.
#'
#' @details
#' If the field "\code{method}" is set to \code{"genoud"}, one can set some parameters of this algorithm:
#'   \code{pop.size}  (default: 50*d),  \code{max.generations} (10*d), \code{wait.generations} (2),  \code{BFGSburnin} (2) and the mutations \code{P1}, \code{P2}, up to \code{P9} (see \code{\link[rgenoud]{genoud}}). Numbers into brackets are the default values.
#' If the field \code{method} is set to \code{"discrete"}, one can set the field \code{optim.points}: p * d matrix corresponding to the p points where the criterion will be evaluated. If nothing is specified, 100*d points are chosen randomly.
#' Finally, one can control the field \code{optim.option} in order to decide how to optimize the sampling criterion.
#' If \code{optim.option} is set to 2 (default), \code{batchsize} sequential optimizations in dimension d are performed to find the optimum.
#' If \code{optim.option} is set to 1, only one optimization in dimension \code{batchsize*d} is performed. This option is only available with \code{"genoud"}. This option might provide more global and accurate solutions, but is a lot more expensive.
#'
#' @examples
#' \donttest{
#' #-------------------------------------------------------------------
#' #------------------------- archissur -------------------------------
#' #-------------------------------------------------------------------
#'
#' #  20-points DoE, and the corresponding response
#' d <- 2
#' nb_PX <- 20
#' x <- matrix(c(0.205293785978832, 0.0159983370750337,
#'               0.684774733109666, 0.125251417595962,
#'               0.787208786290006, 0.700475706055049,
#'               0.480507717105934, 0.359730889653793,
#'               0.543665267336735, 0.565974761807069,
#'               0.303412043992361, 0.471502352650857,
#'               0.839505250127309, 0.504914690245002,
#'               0.573294917143728, 0.784444726564573,
#'               0.291681289223421, 0.255053812451938,
#'               0.87233450888786, 0.947168337730927,
#'               0.648262257638515, 0.973264712407035,
#'               0.421877310273815, 0.0686662506387988,
#'               0.190976166753807, 0.810964668176754,
#'               0.918527262507395, 0.161973686467513,
#'               0.0188128700859558, 0.43522031347403,
#'               0.99902788789426, 0.655561821513544,
#'               0.741113863862512, 0.321050086076934,
#'               0.112003007565305, 0.616551317575545,
#'               0.383511473487687, 0.886611679106771,
#'               0.0749211435982952, 0.205805968972305),
#'             byrow = TRUE, ncol = d)
#'
#' require(DiceKriging)
#' cst_function <- function(z){
#'   fx <- apply(z, 1, branin)
#'   f <- ifelse(fx < 14, 0, 1)
#'   return(f)}
#'
#' ## constraint function
#' s <- cst_function(x)
#'
#' # archissur parameters
#'
#' design.init <- x
#' cst.init <- s
#' n.ite <- 2
#' n_update <- 5
#' lower <- rep(0,d)
#' upper <- rep(1,d)
#' ### GPC model options
#' gpc.options <- list()
#' gpc.options$noise.var <- 1e-6
#' gpc.options$multistart <- 1
#'
#' res <-  archissur(design.init = design.init, cst.init = cst.init,
#'                   cst_function = cst_function, lower = lower, upper = upper,
#'                   n.ite = n.ite, n_update = n_update,  gpc.options = gpc.options)
#' unlink('model_gp.Rds')
#' }
archissur <- function(design.init=NULL, cst.init=NULL, model=NULL, cst_function, lower=NULL, upper=NULL, n.ite=10,
                      seed=NULL, nb.integration=NULL,
                      plot_2D_pn=FALSE, batchsize=1, n_update=1, gpc.options=NULL, optimcontrol=NULL, verbose = 1 ){


  noise.var <- gpc.options$noise.var
  if(is.null(noise.var)){noise.var <- 1e-6}

  normalize <- gpc.options$normalize
  if(is.null(normalize)){normalize <- TRUE}
  
  multistart <- gpc.options$multistart
  if(is.null(multistart)){multistart <- 1}

  MeanTransform <- gpc.options$MeanTransform

  if(!is.null(gpc.options$covtype)){cov_type <- gpc.options$covtype}
  else{cov_type <- "matern5_2"}

  if(is.null(design.init) & is.null(model)){stop("Error: Please provide either an initial DoE or an initial gpcm model to run archissur")}

  if(is.null(design.init)  & !is.null(model)){start="from_model"}
  else{start=TRUE}
  


  if(start==TRUE){ ### start from given init DoE
    if(is.null(design.init)){print("/!\ Initial design of experiment design.init is missing");return(NULL);}
    if(is.null(cst.init)){cst.init <- cst_function(design.init)}
    d <- ncol(design.init)
    cst.init <- cst.init*2 - 1

    obs.cst <- cst.init
    design.cst <- design.init


 



    model <- gpcm(f=obs.cst, Xf=design.cst, covtype=cov_type, noise.var=noise.var, multistart=multistart, normalize=normalize, seed=seed, MeanTransform=MeanTransform)
  }
  else if(start=='from_model') ### get current model in working directory
  {

    print('Loading GPC model')
    model <- readRDS("model_gp.Rds")
    design.cst <- model@X
    obs.cst <- model@y
    cov_type <- model@covariance@name
    d <- model@d
  }

    current.vorob = NULL
    current.vorob.vol = NULL
    current.distance = NULL
    current.alpha = NULL
    model.hyperparam = NULL


  if(is.null(lower)){lower<-rep(0,d)}
  if(is.null(upper)){upper<-rep(1,d)}



  if(is.null(nb.integration)) nb.integration <- d*1000
  integration.points <- sobol(n = nb.integration, dim=d, scrambling = 0)

  integration.param <- list(4)

  integration.param$integration.points <- sweep(rep(upper-lower,each=nb.integration) * matrix(integration.points, nrow=nb.integration),
                                                1,matrix(rep(lower,each=nb.integration), nrow=nb.integration),FUN="+")
  integration.param$integration.weights <- NULL
  integration.param$alpha <- NULL

 



  if(plot_2D_pn & d!=2){warning(paste("impossible to plot pn map in dimension ",as.character(d),sep=""))}
  else{jolie_palette <- function(n) hcl.colors(n, "RdYlBu", rev = FALSE)}


  for(i in 1:n.ite){
    
    if(verbose == 1 || verbose == 2){
    print('-------------------------------------')
    print(paste('iteration: ',as.character(i),sep=""))
    }


    res <- max_vorob_parallel_gpc(lower=lower,upper=upper, batchsize=batchsize, integration.param=integration.param, object=model, optimcontrol=optimcontrol, seed=seed)

    if(plot_2D_pn & d==2){
      ngrid = 35
      x.grid <- seq(0,1, length.out=ngrid)
      grid <- as.matrix(expand.grid(x.grid,x.grid))
      pn_plot<- predict(object=model, newdata=grid, light.return=TRUE)
      filled.contour(x.grid,x.grid, matrix(pn_plot, ngrid, ngrid),
                     color.palette = jolie_palette,
                     main=paste("Pn map - iteration: ", as.character(i), sep=""),
                     plot.axes = {
                       axis(1)
                       axis(2)
                       points(res$par, col = "green", pch = 23, bg = "green",cex = 1.5 )
                       points(design.cst[obs.cst==1,1],design.cst[obs.cst==1,2],col = "blue", pch = 21, bg = "blue")
                       points(design.cst[obs.cst==-1,1],design.cst[obs.cst==-1,2],col = "red", pch = 21, bg = "red")
                     }
      )
    }



    model.hyperparam <-rbind(model.hyperparam, c(model@noise.var,model@coef.m,model@covariance@range.val))

    design.cst <- rbind(design.cst, res$par)
    cst.best <- cst_function(matrix(res$par, ncol =d))*2 - 1
    obs.cst <- c(obs.cst, cst.best)
    if(is.null(res$current.vorob)) current.vorob <- c(current.vorob, NaN)
    if(is.null(res$current.vorob.vol)) current.vorob.vol <- c(current.vorob.vol, NaN)
    current.vorob <- c(current.vorob, res$current.vorob)
    current.vorob.vol <-  c(current.vorob.vol, res$current.vorob.vol)

    current.alpha <- c(current.alpha, res$alpha)
    rownames(design.cst) <- NULL
    rownames(obs.cst) <- NULL
    
    if (verbose == 1) {
      print('-------------------------------------')
      cat("Best point: ", as.matrix(res$par,ncol=d),"\n",sep="\t")
    } 
    else if (verbose == 2) {
      print('-------------------------------------')
      cat("Best point: ", as.matrix(res$par,ncol=d),"\n",sep="\t")
      cat("Criterion value: ", res$value,"\n",sep="\t")
      cat("Constraint value: ",cst.best,"\n",sep="\t")
      cat("Model hyperparameters (mean,theta): ", model.hyperparam[nrow(model.hyperparam),],"\n",sep="\t")
    }
    

    #saveRDS(model, "model_gp.Rds")


    if(i%%n_update==0){
      model <- tryCatch(update(object=model, newf=cst.best, newXf=matrix(res$par, ncol =model@d), covtype=cov_type, multistart=multistart, normalize=normalize, seed=seed),
                        error = function(e){
                          update(object=model, newf=cst.best, newXf=matrix(res$par, ncol =model@d), normalize=normalize, covandmean.reestim=FALSE,seed=seed)
                        })
    }else{
      model<-update(object=model,newf=cst.best, newXf=matrix(res$par, ncol =model@d), normalize=normalize, covandmean.reestim=FALSE, seed=seed)
    }


  }


  model <- tryCatch(update(object=model, newf=cst.best, newXf=matrix(res$par,ncol=model@d), covtype=cov_type, multistart=multistart, normalize=normalize, seed=seed),
                    error = function(e){
                      update(object=model, newf=cst.best, newXf=matrix(res$par, ncol =model@d), normalize=normalize, covandmean.reestim=FALSE, seed=seed)
                    })




  gpc_int <- predict(object=model, newdata=integration.points)
  intpoints.c <- gpc_int$c # unconditional covariance between integration.points and design points
  intpoints.oldmean <- gpc_int$Zsimu_mean
  intpoints.oldsd <- sqrt(gpc_int$Zsimu_var)
  pn <- gpc_int$prob
  alpha <- KrigInv::vorob_threshold(pn)
  
  pn_bigger_than_alpha <- (pn>alpha)+0
  pn_lower_than_alpha <- 1-pn_bigger_than_alpha
  ### current vobol dev and exp comp
  current.vorob.last <- mean(pn*pn_lower_than_alpha + (1-pn)*pn_bigger_than_alpha)


  current.vorob <- c(current.vorob, current.vorob.last)
  current.vorob.vol <-  c(current.vorob.vol,  mean(pn_bigger_than_alpha))
  current.alpha <- c(current.alpha, alpha)
  rownames(design.cst) <- NULL
  rownames(obs.cst) <- NULL
  #saveRDS(model, "model_gp.Rds")

  return(list(Xf=design.cst, f=obs.cst, alpha=res$alpha, vorob_expect=res$current.vorob.vol, vorob_dev=res$current.vorob, model=model))


}
