## ----setup, include = FALSE---------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

## ----message=FALSE------------------------------------------------------------
require(causalBatch)
require(ggplot2)
require(ggpubr)
require(tidyr)
n = 200

## ----eval=FALSE---------------------------------------------------------------
#  vignette("cb.simulations", package="causalBatch")

## -----------------------------------------------------------------------------
# a function for plotting a scatter plot of the data
plot.sim <- function(Ys, Ts, Xs, title="", 
                     xlabel="Covariate",
                     ylabel="Outcome (1st dimension)") {
  data = data.frame(Y1=Ys[,1], Y2=Ys[,2], 
                    Group=factor(Ts, levels=c(0, 1), ordered=TRUE), 
                    Covariates=Xs)
  
  data %>%
    ggplot(aes(x=Covariates, y=Y1, color=Group)) +
    geom_point() +
    labs(title=title, x=xlabel, y=ylabel) +
    scale_x_continuous(limits = c(-1, 1)) +
    scale_color_manual(values=c(`0`="#bb0000", `1`="#0000bb"), 
                       name="Group/Batch") +
    theme_bw()
}

## ----fig.width=5, fig.height=3------------------------------------------------
sim.simpl = cb.sims.sim_sigmoid(n=n, eff_sz=1, unbalancedness=1.5)

plot.sim(sim.simpl$Ys, sim.simpl$Ts, sim.simpl$Xs, title="Sigmoidal Simulation")

## ----eval=FALSE---------------------------------------------------------------
#  vignette("cb.detect.caus_cdcorr", package="causalBatch")

## -----------------------------------------------------------------------------
result <- cb.detect.caus_cdcorr(sim.simpl$Ys, sim.simpl$Ts, sim.simpl$Xs, R=100)
print(sprintf("p-value: %.4f", result$Test$p.value))

## -----------------------------------------------------------------------------
cor.sim.simpl <- cb.correct.aipw_cComBat(sim.simpl$Ys, sim.simpl$Ts, 
                                         data.frame(Covar=sim.simpl$Xs),
                                         "Covar")

## ----fig.width=5, fig.height=3------------------------------------------------
plot.sim(cor.sim.simpl$Ys.corrected, cor.sim.simpl$Ts, cor.sim.simpl$Xs$Covar,
         title="Sigmoidal Simulation (AIPW cComBat corrected)")

## -----------------------------------------------------------------------------
result.cor <- cb.detect.caus_cdcorr(cor.sim.simpl$Ys.corrected, cor.sim.simpl$Ts,
                                    cor.sim.simpl$Xs$Covar, R=100)
print(sprintf("p-value: %.4f", result.cor$Test$p.value))

## -----------------------------------------------------------------------------
Xs.2covar <- data.frame(Covar1=sim.simpl$Xs, Covar2=runif(n))

## -----------------------------------------------------------------------------
cor.sim <- cb.correct.aipw_cComBat(sim.simpl$Ys, sim.simpl$Ts, Xs.2covar, 
                                   "Covar1 + Covar2")

## -----------------------------------------------------------------------------
cor.sim <- cb.correct.aipw_cComBat(sim.simpl$Ys, sim.simpl$Ts, Xs.2covar, 
                                   aipw.form = "Covar1 + Covar2", covar.out.form = "Covar1")

