#' Chronic Virus Infection and IR Model
#'
#' @description A model for a chronic virus infection with a simple immune response
#'
#' @details The model includes uninfected and infected target cells, virus, and the innate and adaptive immune response. The processes that are modeled are cell birth and death, infection, virus production, infected cell and virus death. Immune response dynamics and action are also modeled. See the DSAIRM documentation for model details.
#'
#' This code was generated by the modelbuilder R package.
#' The model is implemented as a set of ordinary differential equations using the deSolve package.
#' The following R packages need to be loaded for the function to work: deSolve.
#'
#' @param U : starting value for Uninfected cells : numeric
#' @param I : starting value for Infected cells : numeric
#' @param V : starting value for Virus : numeric
#' @param F : starting value for Innate Response : numeric
#' @param T : starting value for Adaptive Response : numeric
#' @param n : rate of new uninfected cell replenishment : numeric
#' @param dU : rate at which uninfected cells die : numeric
#' @param dI : rate at which infected cells die : numeric
#' @param dV : rate at which virus is cleared : numeric
#' @param b : rate at which virus infects cells : numeric
#' @param p : rate at which infected cells produce virus : numeric
#' @param g : possible conversion factor for virus units : numeric
#' @param rF : rate of innate response induction : numeric
#' @param dF : rate of innate response decay : numeric
#' @param kF : strength of innate response action : numeric
#' @param rT : rate of adaptive response induction : numeric
#' @param dT : rate of adaptive response decay : numeric
#' @param kT : strength of adaptive response action : numeric
#' @param tstart : Start time of simulation : numeric
#' @param tfinal : Final time of simulation : numeric
#' @param dt : Time step : numeric
#' @return The function returns the output as a list.
#' The time-series from the simulation is returned as a dataframe saved as list element \code{ts}.
#' The \code{ts} dataframe has one column per compartment/variable. The first column is time.
#' @examples
#' # To run the simulation with default parameters:
#' result <- simulate_chronicvirusir_ode()
#' # To choose values other than the standard one, specify them like this:
#' result <- simulate_chronicvirusir_ode(U = 2e+05,I = 0,V = 2,F = 2,T = 2)
#' # You can display or further process the result, like this:
#' plot(result$ts[,'time'],result$ts[,'U'],xlab='Time',ylab='Numbers',type='l')
#' print(paste('Max number of U: ',max(result$ts[,'U'])))
#' @section Warning: This function does not perform any error checking. So if you try to do something nonsensical (e.g. have negative values for parameters), the code will likely abort with an error message.
#' @section Model Author: Andreas Handel
#' @section Model creation date: 2022-05-09
#' @section Code Author: generated by the \code{modelbuilder} R package
#' @section Code creation date: 2022-05-09
#' @export

simulate_chronicvirusir_ode <- function(U = 1e+05, I = 0, V = 10, F = 1, T = 1, n = 1e4, dU = 0.1, dI = 1, dV = 4, b = 1e-06, p = 100, g = 1, rF = 0.001, dF = 1, kF = 0.01, rT = 0.001, dT = 1, kT = 0.01, tstart = 0, tfinal = 200, dt = 0.1)
{
  ##############################
  #Block of ODE equations for deSolve
  ##############################
  Chronic_Virus_Infection_and_IR_Model_ode_fct <- function(t, y, parms)
  {
    with( as.list(c(y,parms)), { #lets us access variables and parameters stored in y and parms by name
    #StartODES
    #Uninfected cells : Uninfected cell birth : Uninfected cell death : Infection of cells :
    dU_mb = +n -dU*U -b*V*U
    #Infected cells : infection of cells : death of infected cells : T-cell killing :
    dI_mb = +b*V*U -dI*I -kT*T*I
    #Virus : virus production : virus removal : infection of cells :
    dV_mb = +p*I/(1+kF*F) -dV*V -b*g*V*U
    #Innate Response : innate response growth : innate response decline :
    dF_mb = +rF*I -dF*F
    #Adaptive Response : adaptive response growth : adaptive response decline :
    dT_mb = +rT*T*V -dT*T
    #EndODES
    list(c(dU_mb,dI_mb,dV_mb,dF_mb,dT_mb))
  } ) } #close with statement, end ODE code block

  ##############################
  #Main function code block
  ##############################
  #Creating named vectors
  varvec_mb = c(U = U, I = I, V = V, F = F, T = T)
  parvec_mb = c(n = n, dU = dU, dI = dI, dV = dV, b = b, p = p, g = g, rF = rF, dF = dF, kF = kF, rT = rT, dT = dT, kT = kT)
  timevec_mb = seq(tstart, tfinal,by = dt)
  #Running the model
  simout = deSolve::ode(y = varvec_mb, parms = parvec_mb, times = timevec_mb,  func = Chronic_Virus_Infection_and_IR_Model_ode_fct, rtol = 1e-12, atol = 1e-12)
  #Setting up empty list and returning result as data frame called ts
  result <- list()
  result$ts <- as.data.frame(simout)
  return(result)
}
